#!/bin/bash -e

# $Id: ttf2tex.sh,v 0.70 2004/09/30 00:00:00 lehman pub $

# Copyright (c) 2000-2002,2004 Philipp Lehman <lehman@gmx.net>
#
# ttf2tex is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any
# later version.
#
# ttf2tex is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place Suite 330, Boston, MA 02111-1307
# USA.

# Internal variables

myname="${0##*/}"
fullname="ttf2tex"
version="v0.7"
released="2004-09-30"
maintainer="<no maintainer>" # email address of maintainer
confpath="./ $HOME/. /etc/"
logfile="${fullname}.log"
date="$(date '+%Y-%m-%d %H:%M:%S')"
texdate="$(date '+%Y/%m/%d')"
debug_cmdline="$0 -b $*"
transcript=0

# Global auxiliary funtions

info() {
  trap error_internal ERR
  echo -e "$myname: $*"
  if [ $transcript = 1 ]; then echo -e "$myname: $*" >>$logfile; fi
}

info_external() {
  trap error_internal ERR
  if [ -n "$2" ]; then echo -e "$1: $2"; fi
  if [ -n "$2" -a $transcript = 1 ]; then echo -e "$1: $2" >>$logfile; fi
}

error() {
  trap error_internal ERR
  if [ -n "$*" ]; then info "$*"; else info "Unknown error"; fi
  info "Aborting..."
  exit 1
}

error_usage() {
  trap error_internal ERR
  usage
  exit 1
}

error_internal() {
  if [ "$DEBUG" = "1" ]; then return; fi
  cat <<-EOF

	An internal error occured! Please send a bug report to $maintainer.
	I will now try to run $fullname again with the same options and debugging
	turned on. The output will be written to the file 'debug.log'. Please
	attach this file to your bug report.

	Please note that 'debug.log' will contain a full dump of the current
	environment. If you set passwords or any other sensitive data in your
	environment, remove the relevant parts from 'debug.log'.
	EOF

  if [ "$confirm" != "0" ]; then
    echo -e "\nPress <enter> to proceed or <ctrl-c> to abort"
    read
  fi
  BASH_OPTIONS="$-"
  export BASH_OPTIONS
  echo -e "+++ ENVIRONMENT +++\n\n$(set)\n\n+++ DEBUG LOG +++\n" >debug.log
  (DEBUG="1"; export DEBUG; bash -ex $debug_cmdline 2>>debug.log)
  echo -e "\nOutput written to debug.log"
  exit 1
}

clean() {
  trap error_internal ERR
  if [ "$DEBUG" = "1" ]; then return; fi
  info "Cleaning up..."
  for file in *.enc ttf2tfm.log ttf2afm.log mktexlsr.log $file_map_ttf2pk $pdftexmap \
	      ${basefamily}*.afm ${basefamily}*.tfm ${basefamily}*.vf ${basefamily}*.fd ${basefamily}*.vpl
  do
    if [ -e $file ]; then rm $file; fi
  done
}

usage() {
  trap error_internal ERR
  cat <<-EOF
	$fullname $version $released
	Usage:
	  $myname --help
	  $myname --dump-vectors
	  $myname [options] --foundry <foundry> --font <font> <family>
	  $myname [options] --user --foundry <foundry> --font <font> <family>
	  $myname [options] --install --foundry <foundry> --font <font> <family>
	Options:
	  -h|--help		 print usage summary and exit
	     --dump-vectors      dump encoding vectors and exit
	  -b|--batch		 run in batch mode
	  -l|--log		 write session log to $logfile
	  -n|--ps-names	 	 pass the -n switch to ttf2tfm
	  -N|--ps-names-only	 pass the -N switch to ttf2tfm
	  -t|--typewriter	 font is a typewriter font, disable hyphenation
	  -x|--expert	         expert mode, use small caps and old style figures
	  -u|--user		 install files into user TeX tree
	  -i|--install		 install files into local TeX tree
	     --overwrite	 overwrite files when installing, use with care!
	  -c|--caps <real>	 use <real> as height of small caps (default: ${capsheight})
	  -s|--slant <real>	 use <real> as obliqueness factor (default: ${slantfactor})
	  -f|--foundry <foundry> use <foundry> as foundry name
	  -o|--font <font>	 use <font> as font name
	  <family>		 the font family code
	EOF
}

# Hash required binaries to check if they are available

check_requirements() {
  trap error_internal ERR

  hash ttf2tfm >/dev/null 2>&1 || {
    info  "Failed to find ttf2tfm!"
    error "Make sure that the freetype tools are installed and that \$PATH is set correctly!"
  }

  hash ttf2afm >/dev/null 2>&1 || {
    info  "Failed to find ttf2afm!"
    error "Make sure that \$PATH is set correctly!"
  }

  hash vptovf >/dev/null 2>&1 || {
    info  "Failed to find vptovf!"
    error "Make sure that \$PATH is set correctly!"
  }
}

# Confirmation prompt

prompt_confirm() {
  trap error_internal ERR
  if [ $confirm = 1 ]; then
    echo
    info "Press <enter> to proceed or <ctrl-c> to abort"
    read
  fi
}

# Display install paths

info_paths() {
  trap error_internal ERR

  if [ -n "$texmfinstall" ]; then
    local install="$texmfinstall"
  else
    local install="$PWD"
  fi
  info "------------------------"
  info "   INSTALLATION PATHS"
  info "------------------------"
  info "Configuration file: $conffile"
  info "Root directory:     $install/"
  info "Font metrics:       $install/$path_afm/$foundry/$fontname/*.afm"
  info "TeX font metrics:   $install/$path_tfm/$foundry/$fontname/*.tfm"
  info "Virtual fonts:      $install/$path_vf/$foundry/$fontname/*.vf"
  info "TrueType fonts:     $install/$path_ttf/$foundry/$fontname/*.ttf"
  info "pdfTeX map files:   $install/$path_map_pdftex/*.map"
  info "pdfTeX enc files:   $install/$path_enc_pdftex/*.enc"
  info "ttf2pk map file:    $install/$path_map_ttf2pk/$file_map_ttf2pk"
  info "ttf2pk enc files:   $install/$path_enc_ttf2pk/*.enc"
  info "Font definitions:   $install/$path_fd/$foundry/$fontname/*.fd"
}

# Check for available weights

check_available() {
  trap error_internal ERR

  # ultra light

  if [ -f "${basefamily}a${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries a"
    mffallback="ul"
  fi

  # extra light

  if [ -f "${basefamily}j${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries j"
    mffallback="el"
  fi

  # light

  if [ -f "${basefamily}l${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries l"
    mffallback="l"
  fi

  # book

  if [ -f "${basefamily}k${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries k"
    mffallback="k"
  fi

  # regular

  if [ -f "${basefamily}r${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries r"
    mffallback="m"
  fi
  
  # poster

  if [ -f "${basefamily}p${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries p"
    bffallback="pb"
    ubdefault="pb"
  fi

  # ultra bold

  if [ -f "${basefamily}u${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries u"
    bffallback="ub"
    ubdefault="ub"
  fi

  # black

  if [ -f "${basefamily}c${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries c"
    bffallback="cb"
    ebdefault="cb"
  fi

  # heavy

  if [ -f "${basefamily}h${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries h"
    bffallback="hb"
    ebdefault="hb"
  fi

  # extra bold

  if [ -f "${basefamily}x${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries x"
    bffallback="eb"
    ebdefault="eb"
  fi

  # medium

  if [ -f "${basefamily}m${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries m"
    bffallback="mb"
    bfdefault=$bffallback
  fi

  # demibold

  if [ -f "${basefamily}d${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries d"
    bffallback="db"
    bfdefault=$bffallback
  fi

  # semibold

  if [ -f "${basefamily}s${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries s"
    bffallback="sb"
    bfdefault=$bffallback
  fi

  # bold

  if [ -f "${basefamily}b${rawenc_wgl}.ttf" ]; then
    fontseries="$fontseries b"
    bffallback="b"
  fi

  # sanity checks

  [ -n "$mffallback" -o -n "$bffallback" ] || error "No font files found!"
  
  if [ -z "$mffallback" ]; then
    mffallback="$bffallback"
  fi

  if [ -z "$bffallback" ]; then
    bffallback="$mffallback"
  fi

  if [ -z "$bfdefault" ]; then
    bfdefault=$bffallback
  fi
}

# Auxiliary functions for info_map_available

info_map_upright() {
  trap error_internal ERR

  local family="$1"
  local toweight="$2"
  local fromweight="$3"
  
  info "${family}/${fromweight}/n  --> ${basefamily}${toweight}${rawenc_wgl}.ttf"
  info "${family}/${fromweight}/sc --> ${basefamily}${toweight}${rawenc_wgl}.ttf"
  info "${family}/${fromweight}/sl --> ${basefamily}${toweight}${rawenc_wgl}.ttf"
}

info_map_italic() {
  trap error_internal ERR

  local family="$1"
  local toweight="$2"
  local fromweight="$3"
  
  info "${family}/${fromweight}/it --> ${basefamily}${toweight}i${rawenc_wgl}.ttf"
}

info_map_substitution() {
  trap error_internal ERR

  local family="$1"
  local toweight="$2"
  local fromweight="$3"
  
  info "[sub] ${family}/${fromweight} --> ${family}/${toweight}"
}

info_map_substseries() {
  trap error_internal ERR

  local family="$1"
  local toweight="$2"
  local fromweight="$3"
  
  info "[sub] ${family}/${fromweight}/n  --> ${family}/${toweight}/n"
  info "[sub] ${family}/${fromweight}/sc --> ${family}/${toweight}/sc"
  info "[sub] ${family}/${fromweight}/sl --> ${family}/${toweight}/sl"
  info "[sub] ${family}/${fromweight}/it --> ${family}/${toweight}/it"
}

# Display available weights

info_map_available() {
  trap error_internal ERR

  info "------------------------"
  info "      FONT MAPPING"
  info "------------------------"

  local family="$1"

  # ultra light

  if [ -f "${basefamily}a${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "a" "ul"
    if [ -f "${basefamily}ai${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "a" "ul"
    else
      info_map_substitution "$family" "ul/sl" "ul/it"
    fi
  fi

  # extra light

  if [ -f "${basefamily}j${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "j" "el"
    if [ -f "${basefamily}ji${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "j" "el"
    else
      info_map_substitution "$family" "el/sl" "el/it"
    fi
  fi

  # light

  if [ -f "${basefamily}l${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "l" "l"
    if [ -f "${basefamily}li${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "l" "l"
    else
      info_map_substitution "$family" "l/sl" "l/it"
    fi
  fi

  # book

  if [ -f "${basefamily}k${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "k" "k"
    if [ -f "${basefamily}ki${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "k" "k"
    else
      info_map_substitution "$family" "k/sl" "k/it"
    fi
  fi

  # regular

  if [ -f "${basefamily}r${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "r" "m"
    if [ -f "${basefamily}ri${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "r" "m"
    else
      info_map_substitution "$family" "m/sl" "m/it"
    fi
  fi
  
  # poster

  if [ -f "${basefamily}p${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "p" "pb"
    if [ -f "${basefamily}pi${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "p" "pb"
    else
      info_map_substitution "$family" "pb/sl" "pb/it"
    fi
  fi

  # ultra bold

  if [ -f "${basefamily}u${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "u" "ub"
    if [ -f "${basefamily}ui${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "u" "ub"
    else
      info_map_substitution "$family" "ub/sl" "ub/it"
    fi
  else
    if [ -n "$ubdefault" ]; then
      info_map_substseries "$family" "$ubdefault" "ub"
    fi
  fi

  # black

  if [ -f "${basefamily}c${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "c" "cb"
    if [ -f "${basefamily}ci${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "c" "cb"
    else
      info_map_substitution "$family" "cb/sl" "cb/it"
    fi
  fi

  # heavy

  if [ -f "${basefamily}h${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "h" "hb"
    if [ -f "${basefamily}hi${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "h" "hb"
    else
      info_map_substitution "$family" "hb/sl" "hb/it"
    fi
  fi

  # extra bold

  if [ -f "${basefamily}x${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "x" "eb"
    if [ -f "${basefamily}xi${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "x" "eb"
    else
      info_map_substitution "$family" "eb/sl" "eb/it"
    fi
  else
    if [ -n "$ebdefault" ]; then
      info_map_substseries "$family" "$ebdefault" "eb"
    fi
  fi

  # medium

  if [ -f "${basefamily}m${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "m" "mb"
    if [ -f "${basefamily}mi${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "m" "mb"
    else
      info_map_substitution "$family" "mb/sl" "mb/it"
    fi
  fi

  # demibold

  if [ -f "${basefamily}d${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "d" "db"
    if [ -f "${basefamily}di${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "d" "db"
    else
      info_map_substitution "$family" "db/sl" "db/it"
    fi
  fi

  # semibold

  if [ -f "${basefamily}s${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "s" "sb"
    if [ -f "${basefamily}si${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "s" "sb"
    else
      info_map_substitution "$family" "sb/sl" "sb/it"
    fi
  fi

  # bold

  if [ -f "${basefamily}b${rawenc_wgl}.ttf" ]; then
    info_map_upright "$family" "b" "b"
    if [ -f "${basefamily}bi${rawenc_wgl}.ttf" ]; then
      info_map_italic "$family" "b" "b"
    else
      info_map_substitution "$family" "b/sl" "b/it"
    fi
  fi

  # substitutions

  if [ -z "$mffallback" ]; then
    info_map_substseries "$family" "$bffallback" "m"
  elif [ "$mffallback" != "m" ]; then
    info_map_substseries "$family" "$mffallback" "m"
  fi

  if [ -z "$bffallback" ]; then
    info_map_substseries "$family" "$mffallback" "b"
  elif [ "-$bffallback" != "-b" ]; then
    info_map_substseries "$family" "$bffallback" "b"
  fi

  info_map_substseries "$family" "$bfdefault" "bx"
}

# Built-in encoding vectors ;)

dump_vectors() {
  trap error_internal ERR

  info "Creating encoding vector T1-WGL4.enc"
  cat <<-EOF >T1-WGL4.enc
	% T1-WGL4.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version. It is based on the T1-WGL4.enc file
	%       distributed with FreeType versions 1.3.x and 1.4.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	% Derived from T1-WGL4.enc by Philipp Lehman <lehman@gmx.net>
	%
	% Version 1.1  2002-05-20
	%
	% LaTeX T1 mapping for Unicode encoded TrueType fonts (WGL4)
	%
	% Note that /hyphen appears twice: at the T1 code points
	% 0x2d (hyphen) and 0x7f (hyphenchar)
	%
	% LIGKERN space l =: lslash ;
	% LIGKERN space L =: Lslash ;
	% LIGKERN question quoteleft =: questiondown ;
	% LIGKERN exclam quoteleft =: exclamdown ;
	% LIGKERN hyphen hyphen =: endash ;
	% LIGKERN endash hyphen =: emdash ;
	% LIGKERN quoteleft quoteleft =: quotedblleft ;
	% LIGKERN quoteright quoteright =: quotedblright ;
	% LIGKERN comma comma =: quotedblbase ;
	% LIGKERN less less =: guillemotleft ;
	% LIGKERN greater greater =: guillemotright ;
	%
	% LIGKERN f i =: fi ;
	% LIGKERN f l =: fl ;
	% LIGKERN f f =: ff ;
	% LIGKERN ff i =: ffi ;
	% LIGKERN ff l =: ffl ;
	%
	% Remove kerns to and from spaces and remove any kerns from
	% the numbers.
	%
	% LIGKERN space {} * ; * {} space ;
	% LIGKERN zero {} * ; * {} zero ;
	% LIGKERN one {} * ; * {} one ;
	% LIGKERN two {} * ; * {} two ;
	% LIGKERN three {} * ; * {} three ;
	% LIGKERN four {} * ; * {} four ;
	% LIGKERN five {} * ; * {} five ;
	% LIGKERN six {} * ; * {} six ;
	% LIGKERN seven {} * ; * {} seven ;
	% LIGKERN eight {} * ; * {} eight ;
	% LIGKERN nine {} * ; * {} nine ;

	/T1Encoding [          % now 256 chars follow
	% 0x00
	  /grave /acute /circumflex /tilde
	  /dieresis /hungarumlaut /ring /caron
	  /breve /macron /dotaccent /cedilla
	  /ogonek /quotesinglbase /guilsinglleft /guilsinglright
	% 0x10
	  /quotedblleft /quotedblright /quotedblbase /guillemotleft
	  /guillemotright /endash /emdash /compwordmark
	  /perthousandzero /dotlessi /dotlessj /ff
	  /fi /fl /ffi /ffl
	% 0x20
	  /visualspace /exclam /quotedbl /numbersign
	  /dollar /percent /ampersand /quoteright
	  /parenleft /parenright /asterisk /plus
	  /comma /hyphen /period /slash
	% 0x30
	  /zero /one /two /three
	  /four /five /six /seven
	  /eight /nine /colon /semicolon
	  /less /equal /greater /question
	% 0x40
	  /at /A /B /C
	  /D /E /F /G
	  /H /I /J /K
	  /L /M /N /O
	% 0x50
	  /P /Q /R /S
	  /T /U /V /W
	  /X /Y /Z /bracketleft
	  /backslash /bracketright /asciicircum /underscore
	% 0x60
	  /quoteleft /a /b /c
	  /d /e /f /g
	  /h /i /j /k
	  /l /m /n /o
	% 0x70
	  /p /q /r /s
	  /t /u /v /w
	  /x /y /z /braceleft
	  /bar /braceright /asciitilde /hyphen
	% 0x80
	  /Abreve /Aogonek /Cacute /Ccaron
	  /Dcaron /Ecaron /Eogonek /Gbreve
	  /Lacute /Lcaron /Lslash /Nacute
	  /Ncaron /Eng /Odblacute /Racute
	% 0x90
	  /Rcaron /Sacute /Scaron /Scedilla
	  /Tcaron /Tcedilla /Udblacute /Uring
	  /Ydieresis /Zacute /Zcaron /Zdotaccent
	  /IJ /Idot /dmacron /section
	% 0xA0
	  /abreve /aogonek /cacute /ccaron
	  /dcaron /ecaron /eogonek /gbreve
	  /lacute /lcaron /lslash /nacute
	  /ncaron /eng /odblacute /racute
	% 0xB0
	  /rcaron /sacute /scaron /scedilla
	  /tcaron /tcedilla /udblacute /uring
	  /ydieresis /zacute /zcaron /zdotaccent
	  /ij /exclamdown /questiondown /sterling
	% 0xC0
	  /Agrave /Aacute /Acircumflex /Atilde
	  /Adieresis /Aring /AE /Ccedilla
	  /Egrave /Eacute /Ecircumflex /Edieresis
	  /Igrave /Iacute /Icircumflex /Idieresis
	% 0xD0
	  /Eth /Ntilde /Ograve /Oacute
	  /Ocircumflex /Otilde /Odieresis /OE
	  /Oslash /Ugrave /Uacute /Ucircumflex
	  /Udieresis /Yacute /Thorn /Germandbls
	% 0xE0
	  /agrave /aacute /acircumflex /atilde
	  /adieresis /aring /ae /ccedilla
	  /egrave /eacute /ecircumflex /edieresis
	  /igrave /iacute /icircumflex /idieresis
	% 0xF0
	  /eth /ntilde /ograve /oacute
	  /ocircumflex /otilde /odieresis /oe
	  /oslash /ugrave /uacute /ucircumflex
	  /udieresis /yacute /thorn /germandbls
	] def

	% eof
	EOF

  info "Creating encoding vector T1-OSF.enc"
  cat <<-EOF >T1-OSF.enc
	% T1-OSF.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version. It is based on the T1-WGL4.enc file
	%       distributed with FreeType versions 1.3.x and 1.4.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	% Derived from T1-WGL4.enc by Philipp Lehman <lehman@gmx.net>
	%
	% Version 1.1  2002-05-20
	%
	% LaTeX T1 mapping for Unicode encoded TrueType fonts (WGL4),
	% using old style instead of the default figures
	%
	% Note that /hyphen appears twice: at the T1 code points
	% 0x2d (hyphen) and 0x7f (hyphenchar)
	%
	% LIGKERN space l =: lslash ;
	% LIGKERN space L =: Lslash ;
	% LIGKERN question quoteleft =: questiondown ;
	% LIGKERN exclam quoteleft =: exclamdown ;
	% LIGKERN hyphen hyphen =: endash ;
	% LIGKERN endash hyphen =: emdash ;
	% LIGKERN quoteleft quoteleft =: quotedblleft ;
	% LIGKERN quoteright quoteright =: quotedblright ;
	% LIGKERN comma comma =: quotedblbase ;
	% LIGKERN less less =: guillemotleft ;
	% LIGKERN greater greater =: guillemotright ;
	%
	% LIGKERN f i =: fi ;
	% LIGKERN f l =: fl ;
	% LIGKERN f f =: ff ;
	% LIGKERN ff i =: ffi ;
	% LIGKERN ff l =: ffl ;
	%
	% Remove kerns to and from spaces and remove any kerns from
	% the numbers.
	%
	% LIGKERN space {} * ; * {} space ;
	% LIGKERN zerooldstyle {} * ; * {} zerooldstyle ;
	% LIGKERN oneoldstyle {} * ; * {} oneoldstyle ;
	% LIGKERN twooldstyle {} * ; * {} twooldstyle ;
	% LIGKERN threeoldstyle {} * ; * {} threeoldstyle ;
	% LIGKERN fouroldstyle {} * ; * {} fouroldstyle ;
	% LIGKERN fiveoldstyle {} * ; * {} fiveoldstyle ;
	% LIGKERN sixoldstyle {} * ; * {} sixoldstyle ;
	% LIGKERN sevenoldstyle {} * ; * {} sevenoldstyle ;
	% LIGKERN eightoldstyle {} * ; * {} eightoldstyle ;
	% LIGKERN nineoldstyle {} * ; * {} nineoldstyle ;

	/T1Encoding [          % now 256 chars follow
	% 0x00
	  /grave /acute /circumflex /tilde
	  /dieresis /hungarumlaut /ring /caron
	  /breve /macron /dotaccent /cedilla
	  /ogonek /quotesinglbase /guilsinglleft /guilsinglright
	% 0x10
	  /quotedblleft /quotedblright /quotedblbase /guillemotleft
	  /guillemotright /endash /emdash /compwordmark
	  /perthousandzero /dotlessi /dotlessj /ff
	  /fi /fl /ffi /ffl
	% 0x20
	  /visualspace /exclam /quotedbl /numbersign
	  /dollar /percent /ampersand /quoteright
	  /parenleft /parenright /asterisk /plus
	  /comma /hyphen /period /slash
	% 0x30
	  /zerooldstyle /oneoldstyle /twooldstyle /threeoldstyle
	  /fouroldstyle /fiveoldstyle /sixoldstyle /sevenoldstyle
	  /eightoldstyle /nineoldstyle /colon /semicolon
	  /less /equal /greater /question
	% 0x40
	  /at /A /B /C
	  /D /E /F /G
	  /H /I /J /K
	  /L /M /N /O
	% 0x50
	  /P /Q /R /S
	  /T /U /V /W
	  /X /Y /Z /bracketleft
	  /backslash /bracketright /asciicircum /underscore
	% 0x60
	  /quoteleft /a /b /c
	  /d /e /f /g
	  /h /i /j /k
	  /l /m /n /o
	% 0x70
	  /p /q /r /s
	  /t /u /v /w
	  /x /y /z /braceleft
	  /bar /braceright /asciitilde /hyphen
	% 0x80
	  /Abreve /Aogonek /Cacute /Ccaron
	  /Dcaron /Ecaron /Eogonek /Gbreve
	  /Lacute /Lcaron /Lslash /Nacute
	  /Ncaron /Eng /Odblacute /Racute
	% 0x90
	  /Rcaron /Sacute /Scaron /Scedilla
	  /Tcaron /Tcedilla /Udblacute /Uring
	  /Ydieresis /Zacute /Zcaron /Zdotaccent
	  /IJ /Idot /dmacron /section
	% 0xA0
	  /abreve /aogonek /cacute /ccaron
	  /dcaron /ecaron /eogonek /gbreve
	  /lacute /lcaron /lslash /nacute
	  /ncaron /eng /odblacute /racute
	% 0xB0
	  /rcaron /sacute /scaron /scedilla
	  /tcaron /tcedilla /udblacute /uring
	  /ydieresis /zacute /zcaron /zdotaccent
	  /ij /exclamdown /questiondown /sterling
	% 0xC0
	  /Agrave /Aacute /Acircumflex /Atilde
	  /Adieresis /Aring /AE /Ccedilla
	  /Egrave /Eacute /Ecircumflex /Edieresis
	  /Igrave /Iacute /Icircumflex /Idieresis
	% 0xD0
	  /Eth /Ntilde /Ograve /Oacute
	  /Ocircumflex /Otilde /Odieresis /OE
	  /Oslash /Ugrave /Uacute /Ucircumflex
	  /Udieresis /Yacute /Thorn /Germandbls
	% 0xE0
	  /agrave /aacute /acircumflex /atilde
	  /adieresis /aring /ae /ccedilla
	  /egrave /eacute /ecircumflex /edieresis
	  /igrave /iacute /icircumflex /idieresis
	% 0xF0
	  /eth /ntilde /ograve /oacute
	  /ocircumflex /otilde /odieresis /oe
	  /oslash /ugrave /uacute /ucircumflex
	  /udieresis /yacute /thorn /germandbls
	] def

	% eof
	EOF

  info "Creating encoding vector T1-SC.enc"
  cat <<-EOF >T1-SC.enc
	% T1-SC.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version. It is based on the T1-WGL4.enc file
	%       distributed with FreeType versions 1.3.x and 1.4.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	% Derived from T1-WGL4.enc by Bruce D\'Arcus <darcusb@muohio.edu>
	% Modified and enhanced by Philipp Lehman <lehman@gmx.net>
	%
	% Version 1.3  2002-05-20
	%
	% LaTeX T1 mapping for Unicode encoded TrueType fonts (WGL4),
	% using small caps instead of lowercase glyphs
	%
	% Note that /hyphen appears twice: at the T1 code points
	% 0x2d (hyphen) and 0x7f (hyphenchar)
	%
	% LIGKERN space Lsmall =: Lslashsmall ;
	% LIGKERN space L =: Lslash ;
	% LIGKERN question quoteleft =: questiondown ;
	% LIGKERN exclam quoteleft =: exclamdown ;
	% LIGKERN hyphen hyphen =: endash ;
	% LIGKERN endash hyphen =: emdash ;
	% LIGKERN quoteleft quoteleft =: quotedblleft ;
	% LIGKERN quoteright quoteright =: quotedblright ;
	% LIGKERN comma comma =: quotedblbase ;
	% LIGKERN less less =: guillemotleft ;
	% LIGKERN greater greater =: guillemotright ;
	%
	% Remove kerns to and from spaces and remove any kerns from
	% the numbers.
	%
	% LIGKERN space {} * ; * {} space ;
	% LIGKERN zero {} * ; * {} zero ;
	% LIGKERN one {} * ; * {} one ;
	% LIGKERN two {} * ; * {} two ;
	% LIGKERN three {} * ; * {} three ;
	% LIGKERN four {} * ; * {} four ;
	% LIGKERN five {} * ; * {} five ;
	% LIGKERN six {} * ; * {} six ;
	% LIGKERN seven {} * ; * {} seven ;
	% LIGKERN eight {} * ; * {} eight ;
	% LIGKERN nine {} * ; * {} nine ;

	/T1Encoding [          % now 256 chars follow
	% 0x00
	  /grave /acute /circumflex /tilde
	  /dieresis /hungarumlaut /ring /caron
	  /breve /macron /dotaccent /cedilla
	  /ogonek /quotesinglbase /guilsinglleft /guilsinglright
	% 0x10
	  /quotedblleft /quotedblright /quotedblbase /guillemotleft
	  /guillemotright /endash /emdash /compwordmark
	  /perthousandzero /dotlessIsmall /dotlessJsmall /ff
	  /fi /fl /ffi /ffl
	% 0x20
	  /visualspace /exclam /quotedbl /numbersign
	  /dollaroldstyle /percentsmall /ampersandsmall /quoteright
	  /parenleft /parenright /asterisk /plus
	  /comma /hyphen /period /slash
	% 0x30
	  /zero /one /two /three
	  /four /five /six /seven
	  /eight /nine /colon /semicolon
	  /less /equal /greater /question
	% 0x40
	  /at /A /B /C
	  /D /E /F /G
	  /H /I /J /K
	  /L /M /N /O
	% 0x50
	  /P /Q /R /S
	  /T /U /V /W
	  /X /Y /Z /bracketleft
	  /backslash /bracketright /asciicircum /underscore
	% 0x60
	  /quoteleft /Asmall /Bsmall /Csmall
	  /Dsmall /Esmall /Fsmall /Gsmall
	  /Hsmall /Ismall /Jsmall /Ksmall
	  /Lsmall /Msmall /Nsmall /Osmall
	% 0x70
	  /Psmall /Qsmall /Rsmall /Ssmall
	  /Tsmall /Usmall /Vsmall /Wsmall
	  /Xsmall /Ysmall /Zsmall /braceleft
	  /bar /braceright /asciitilde /hyphen
	% 0x80
	  /Abreve /Aogonek /Cacute /Ccaron
	  /Dcaron /Ecaron /Eogonek /Gbreve
	  /Lacute /Lcaron /Lslash /Nacute
	  /Ncaron /Eng /Odblacute /Racute
	% 0x90
	  /Rcaron /Sacute /Scaron /Scedilla
	  /Tcaron /Tcedilla /Udblacute /Uring
	  /Ydieresis /Zacute /Zcaron /Zdotaccent
	  /IJ /Idot /Dmacronsmall /section
	% 0xA0
	  /Abrevesmall /Aogoneksmall /Cacutesmall /Ccaronsmall
	  /Dcaronsmall /Ecaronsmall /Eogoneksmall /Gbrevesmall
	  /Lacutesmall /Lcaronsmall /Lslashsmall /Nacutesmall
	  /Ncaronsmall /Engsmall /Odblacutesmall /Racutesmall
	% 0xB0
	  /Rcaronsmall /Sacutesmall /Scaronsmall /Scedillasmall
	  /Tcaronsmall /Tcedillasmall /Udblacutesmall /Uringsmall
	  /Ydieresissmall /Zacutesmall /Zcaronsmall /Zdotaccentsmall
	  /IJsmall /exclamdown /questiondown /sterling
	% 0xC0
	  /Agrave /Aacute /Acircumflex /Atilde
	  /Adieresis /Aring /AE /Ccedilla
	  /Egrave /Eacute /Ecircumflex /Edieresis
	  /Igrave /Iacute /Icircumflex /Idieresis
	% 0xD0
	  /Eth /Ntilde /Ograve /Oacute
	  /Ocircumflex /Otilde /Odieresis /OE
	  /Oslash /Ugrave /Uacute /Ucircumflex
	  /Udieresis /Yacute /Thorn /Germandbls
	% 0xE0
	  /Agravesmall /Aacutesmall /Acircumflexsmall /Atildesmall
	  /Adieresissmall /Aringsmall /AEsmall /Ccedillasmall
	  /Egravesmall /Eacutesmall /Ecircumflexsmall /Edieresissmall
	  /Igravesmall /Iacutesmall /Icircumflexsmall /Idieresissmall
	% 0xF0
	  /Ethsmall /Ntildesmall /Ogravesmall /Oacutesmall
	  /Ocircumflexsmall /Otildesmall /Odieresissmall /OEsmall
	  /Oslashsmall /Ugravesmall /Uacutesmall /Ucircumflexsmall
	  /Udieresissmall /Yacutesmall /Thornsmall /germandbls
	] def

	% eof
	EOF

  info "Creating encoding vector T1-SCOSF.enc"
  cat <<-EOF >T1-SCOSF.enc
	% T1-SCOSF.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version. It is based on the T1-WGL4.enc file
	%       distributed with FreeType versions 1.3.x and 1.4.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	% Derived from T1-SC.enc by Philipp Lehman <lehman@gmx.net>
	%
	% Version 1.3  2002-05-20
	%
	% LaTeX T1 mapping for Unicode encoded TrueType fonts (WGL4),
	% using small caps instead of lowercase glyphs and old style
	% instead of the default figures
	%
	% Note that /hyphen appears twice: at the T1 code points
	% 0x2d (hyphen) and 0x7f (hyphenchar)
	%
	% LIGKERN space Lsmall =: Lslashsmall ;
	% LIGKERN space L =: Lslash ;
	% LIGKERN questionsmall quoteleft =: questiondownsmall ;
	% LIGKERN exclamsmall quoteleft =: exclamdownsmall ;
	% LIGKERN hyphen hyphen =: endash ;
	% LIGKERN endash hyphen =: emdash ;
	% LIGKERN quoteleft quoteleft =: quotedblleft ;
	% LIGKERN quoteright quoteright =: quotedblright ;
	% LIGKERN comma comma =: quotedblbase ;
	% LIGKERN less less =: guillemotleft ;
	% LIGKERN greater greater =: guillemotright ;
	%
	% Remove kerns to and from spaces and remove any kerns from
	% the numbers.
	%
	% LIGKERN space {} * ; * {} space ;
	% LIGKERN zerooldstyle {} * ; * {} zerooldstyle ;
	% LIGKERN oneoldstyle {} * ; * {} oneoldstyle ;
	% LIGKERN twooldstyle {} * ; * {} twooldstyle ;
	% LIGKERN threeoldstyle {} * ; * {} threeoldstyle ;
	% LIGKERN fouroldstyle {} * ; * {} fouroldstyle ;
	% LIGKERN fiveoldstyle {} * ; * {} fiveoldstyle ;
	% LIGKERN sixoldstyle {} * ; * {} sixoldstyle ;
	% LIGKERN sevenoldstyle {} * ; * {} sevenoldstyle ;
	% LIGKERN eightoldstyle {} * ; * {} eightoldstyle ;
	% LIGKERN nineoldstyle {} * ; * {} nineoldstyle ;

	/T1Encoding [          % now 256 chars follow
	% 0x00
	  /grave /acute /circumflex /tilde
	  /dieresis /hungarumlaut /ring /caron
	  /breve /macron /dotaccent /cedilla
	  /ogonek /quotesinglbase /guilsinglleft /guilsinglright
	% 0x10
	  /quotedblleft /quotedblright /quotedblbase /guillemotleft
	  /guillemotright /endash /emdash /compwordmark
	  /perthousandzero /dotlessIsmall /dotlessJsmall /ff
	  /fi /fl /ffi /ffl
	% 0x20
	  /visualspace /exclam /quotedbl /numbersign
	  /dollaroldstyle /percentsmall /ampersandsmall /quoteright
	  /parenleft /parenright /asterisk /plus
	  /comma /hyphen /period /slash
	% 0x30
	  /zerooldstyle /oneoldstyle /twooldstyle /threeoldstyle
	  /fouroldstyle /fiveoldstyle /sixoldstyle /sevenoldstyle
	  /eightoldstyle /nineoldstyle /colon /semicolon
	  /less /equal /greater /question
	% 0x40
	  /at /A /B /C
	  /D /E /F /G
	  /H /I /J /K
	  /L /M /N /O
	% 0x50
	  /P /Q /R /S
	  /T /U /V /W
	  /X /Y /Z /bracketleft
	  /backslash /bracketright /asciicircum /underscore
	% 0x60
	  /quoteleft /Asmall /Bsmall /Csmall
	  /Dsmall /Esmall /Fsmall /Gsmall
	  /Hsmall /Ismall /Jsmall /Ksmall
	  /Lsmall /Msmall /Nsmall /Osmall
	% 0x70
	  /Psmall /Qsmall /Rsmall /Ssmall
	  /Tsmall /Usmall /Vsmall /Wsmall
	  /Xsmall /Ysmall /Zsmall /braceleft
	  /bar /braceright /asciitilde /hyphen
	% 0x80
	  /Abreve /Aogonek /Cacute /Ccaron
	  /Dcaron /Ecaron /Eogonek /Gbreve
	  /Lacute /Lcaron /Lslash /Nacute
	  /Ncaron /Eng /Odblacute /Racute
	% 0x90
	  /Rcaron /Sacute /Scaron /Scedilla
	  /Tcaron /Tcedilla /Udblacute /Uring
	  /Ydieresis /Zacute /Zcaron /Zdotaccent
	  /IJ /Idot /Dmacronsmall /section
	% 0xA0
	  /Abrevesmall /Aogoneksmall /Cacutesmall /Ccaronsmall
	  /Dcaronsmall /Ecaronsmall /Eogoneksmall /Gbrevesmall
	  /Lacutesmall /Lcaronsmall /Lslashsmall /Nacutesmall
	  /Ncaronsmall /Engsmall /Odblacutesmall /Racutesmall
	% 0xB0
	  /Rcaronsmall /Sacutesmall /Scaronsmall /Scedillasmall
	  /Tcaronsmall /Tcedillasmall /Udblacutesmall /Uringsmall
	  /Ydieresissmall /Zacutesmall /Zcaronsmall /Zdotaccentsmall
	  /IJsmall /exclamdown /questiondown /sterling
	% 0xC0
	  /Agrave /Aacute /Acircumflex /Atilde
	  /Adieresis /Aring /AE /Ccedilla
	  /Egrave /Eacute /Ecircumflex /Edieresis
	  /Igrave /Iacute /Icircumflex /Idieresis
	% 0xD0
	  /Eth /Ntilde /Ograve /Oacute
	  /Ocircumflex /Otilde /Odieresis /OE
	  /Oslash /Ugrave /Uacute /Ucircumflex
	  /Udieresis /Yacute /Thorn /Germandbls
	% 0xE0
	  /Agravesmall /Aacutesmall /Acircumflexsmall /Atildesmall
	  /Adieresissmall /Aringsmall /AEsmall /Ccedillasmall
	  /Egravesmall /Eacutesmall /Ecircumflexsmall /Edieresissmall
	  /Igravesmall /Iacutesmall /Icircumflexsmall /Idieresissmall
	% 0xF0
	  /Ethsmall /Ntildesmall /Ogravesmall /Oacutesmall
	  /Ocircumflexsmall /Otildesmall /Odieresissmall /OEsmall
	  /Oslashsmall /Ugravesmall /Uacutesmall /Ucircumflexsmall
	  /Udieresissmall /Yacutesmall /Thornsmall /germandbls
	] def

	% eof
	EOF

  info "Creating encoding vector TS1-AGL.enc"
  cat <<-EOF >TS1-AGL.enc
	% TS1-AGL.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	%       This file is still more or less experimental.
	%
	% Originally written by D. van Dok <d.vandoc@ecn.nl>
	% Modified and enhanced by Werner Lemberg <wl@gnu.org>
	%
	% Version 0.1  2001-02-15
	%
	% Glyph names follow the Adobe Glyph List (AGL); WGL4 glyph
	% name variants (mostly for older fonts) as documented in
	% the TrueType specification are as follows.
	%
	%   Omega          -> Ohm
	%   lira           -> afii08941
	%   mu             -> mu1
	%   periodcentered -> middot
	%
	% Glyph names marked with "?" are uncertain; any improvements welcome

	/TS1Encoding [          % now 256 chars follow
	% 0x00
	  /Grave                % CAPITAL GRAVE ACCENT (AGL)
	  /Acute                % CAPITAL ACUTE ACCENT (AGL)
	  /.notdef              % CAPITAL CIRCUMFLEX ACCENT
	  /.notdef              % CAPITAL TILDE ACCENT
	  /Dieresis             % CAPITAL DIAERESIS (AGL)
	  /Hungarumlaut         % CAPITAL DOUBLE ACUTE ACCENT (AGL)
	  /.notdef              % CAPITAL RING ACCENT
	  /Caron                % CAPITAL CARON (AGL)
	  /.notdef              % CAPITAL BREVE ACCENT
	  /Macron               % CAPITAL MACRON (AGL)
	  /.notdef              % CAPITAL DOT ABOVE ACCENT
	  /cedilla              % CAPITAL CEDILLA (AGL, WGL4)
	  /ogonek               % CAPITAL OGONEK (AGL, WGL4)
	  /quotesinglbase       % SINGLE LOW-9 QUOTATION MARK (AGL, WGL4)
	  /.notdef
	  /.notdef
	% 0x10
	  /.notdef
	  /.notdef
	  /quotedblbase         % DOUBLE LOW-9 QUOTATION MARK (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef              % 2/3 em dash
	  /threequartersemdash  % THREE QUARTERS EM DASH (AGL)
	  /.notdef              % capital compound word mark
	  /arrowleft            % LEFTWARDS ARROW (AGL, WGL4)
	  /arrowright           % RIGHTWARDS ARROW (AGL, WGL4)
	  /uni2040              % ? TIE ACCENT (AGL)
	  /.notdef              % ? CAPITAL TIE ACCENT
	  /.notdef              % ? TIE ACCENT (variant)
	  /.notdef              % ? CAPITAL TIE ACCENT (variant)
	  /.notdef
	  /.notdef              % lowercase ascender compound word mark
	% 0x20
	  /uni0180              % LATIN SMALL LETTER B WITH STROKE (AGL)
	  /.notdef
	  /.notdef
	  /.notdef
	  /dollar               % DOLLAR SIGN (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /quotesingle          % APOSTROPHE (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /asterisk             % ASTERISK (AGL, WGL4)
	  /.notdef
	  /comma                % COMMA (AGL, WGL4)
	  /.notdef              % HYPHEN (glyph variant with two strokes)
	  /period               % FULL STOP (AGL, WGL4)
	  /fraction             % ? FRACTION SLASH (AGL, WGL4) or "/slash"
	% 0x30
	  /zerooldstyle         % OLDSTYLE DIGIT ZERO (AGL)
	  /oneoldstyle          % OLDSTYLE DIGIT ONE (AGL)
	  /twooldstyle          % OLDSTYLE DIGIT TWO (AGL)
	  /threeoldstyle        % OLDSTYLE DIGIT THREE (AGL)
	  /fouroldstyle         % OLDSTYLE DIGIT FOUR (AGL)
	  /fiveoldstyle         % OLDSTYLE DIGIT FIVE (AGL)
	  /sixoldstyle          % OLDSTYLE DIGIT SIX (AGL)
	  /sevenoldstyle        % OLDSTYLE DIGIT SEVEN (AGL)
	  /eightoldstyle        % OLDSTYLE DIGIT EIGHT (AGL)
	  /nineoldstyle         % OLDSTYLE DIGIT NINE (AGL)
	  /.notdef
	  /.notdef
	  /angleleft            % LEFT-POINTING ANGLE BRACKET (AGL)
	  /minus                % MINUS SIGN (AGL, WGL4)
	  /angleright           % RIGHT-POINTING ANGLE BRACKET (AGL)
	  /.notdef
	% 0x40
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /uni2127              % INVERTED OHM SIGN (AGL)
	  /.notdef
	  /circle               % WHITE CIRCLE (AGL, WGL4)
	% 0x50
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /Omega                % OHM SIGN (AGL, WGL4 -> "/Ohm")
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef              % OPENING DOUBLE SQUARE BRACKET
	  /.notdef
	  /.notdef              % CLOSING DOUBLE SQUARE BRACKET
	  /arrowup              % UPWARDS ARROW (AGL, WGL4)
	  /arrowdown            % DOWNWARDS ARROW (AGL, WGL4)
	% 0x60
	  /grave                % GRAVE ACCENT (AGL, WGL4)
	  /.notdef
	  /.notdef              % BORN
	  /.notdef              % DIVORCED
	  /.notdef              % DIED
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef              % LEAF
	  /.notdef              % MARRIED
	  /musicalnote          % EIGTH NOTE (AGL, WGL4)
	  /.notdef
	% 0x70
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /asciitilde           % TILDE (AGL, WGL4)
	  /.notdef              % HYPHEN (glyph variant with two strokes)
	% 0x80
	  /breve                % BREVE (AGL, WGL4)
	  /caron                % CARON (AGL, WGL4)
	  /hungarumlaut         % DOUBLE ACUTE ACCENT (AGL, WGL4)
	  /dblGrave             % CAPITAL GRAVE DOUBLE ACCENT (AGL)
	  /dagger               % DAGGER (AGL, WGL4)
	  /daggerdbl            % DOUBLE DAGGER (AGL, WGL4)
	  /.notdef              % double vertical bar
	  /perthousand          % PER MILLE SIGN (AGL, WGL4)
	  /bullet               % BULLET (AGL, WGL4)
	  /uni2103              % DEGREE CELSIUS (AGL)
	  /dollaroldstyle       % OLDSTYLE DOLLAR SIGN (AGL)
	  /centoldstyle         % OLDSTYLE CENT SIGN (AGL)
	  /florin               % LATIN SMALL LETTER F WITH HOOK (AGL, WGL4)
	  /colonmonetary        % COLON SIGN (AGL)
	  /uni20A9              % WON SIGN (AGL)
	  /uni20A6              % NAIRA SIGN (AGL)
	% 0x90
	  /.notdef              % ? LATIN CAPITAL LETTER G WITH VERTICAL STROKE (guarani)
	  /.notdef              % ? LATIN CAPITAL LETTER P WITH STROKE (peso)
	  /lira                 % LIRA SIGN (AGL, WGL4 -> "/afii08941")
	  /prescription         % ? RECIPE (AGL)
	  /uni203D              % INTERROBANG (AGL)
	  /.notdef              % INTERROBANG INVERTED
	  /dong                 % DONG SIGN (AGL)
	  /trademark            % TRADE MARK SIGN (AGL, WGL4)
	  /uni2031              % PER TEN THOUSAND SIGN (AGL)
	  /.notdef              % ? PILCROW SIGN (glyph variant)
	  /uni0E3F              % THAI CURRENCY SYMBOL BAHT (AGL)
	  /afii61352            % NUMERO SIGN (AGL, WGL4)
	  /.notdef              % PERCENT SIGN (glyph variant)
	  /estimated            % ESTIMATED SYMBOL (AGL, WGL4)
	  /openbullet           % WHITE BULLET (AGL, WGL4)
	  /uni2120              % SERVICE MARK (AGL)
	% 0xA0
	  /uni2045              % LEFT SQUARE BRACKET WITH SQUILL (AGL)
	  /uni2046              % RIGHT SQUARE BRACKET WITH SQUILL (AGL)
	  /cent                 % CENT SIGN (AGL, WGL4)
	  /sterling             % POUND SIGN (AGL, WGL4)
	  /currency             % CURRENCY SIGN (AGL, WGL4)
	  /yen                  % YEN SIGN (AGL, WGL4)
	  /brokenbar            % BROKEN BAR (AGL, WGL4)
	  /section              % SECTION SIGN (AGL, WGL4)
	  /dieresis             % DIAERESIS (AGL, WGL4)
	  /copyright            % COPYRIGHT SIGN (AGL, WGL4)
	  /ordfeminine          % FEMININE ORDINAL INDICATOR (AGL, WGL4)
	  /.notdef              % COPYLEFT SIGN
	  /logicalnot           % NOT SIGN (AGL, WGL4)
	  /uni2117              % SOUND RECORDING COPYRIGHT (AGL)
	  /registered           % REGISTERED SIGN (AGL, WGL4)
	  /macron               % MACRON (AGL, WGL4)
	% 0xB0
	  /degree               % DEGREE SIGN (AGL, WGL4)
	  /plusminus            % PLUS-MINUS SIGN (AGL, WGL4)
	  /twosuperior          % SUPERSCRIPT TWO (AGL, WGL4)
	  /threesuperior        % SUPERSCRIPT THREE (AGL, WGL4)
	  /acute                % ACUTE ACCENT (AGL, WGL4)
	  /mu                   % MICRO SIGN (AGL, WGL4 -> "/mu1")
	  /paragraph            % PILCROW SIGN (AGL, WGL4)
	  /periodcentered       % MIDDLE DOT (AGL, WGL4 -> "/middot")
	  /uni203B              % REFERENCE MARK
	  /onesuperior          % SUPERSCRIPT ONE (AGL, WGL4)
	  /ordmasculine         % MASCULINE ORDINAL INDICATOR (AGL, WGL4)
	  /radical              % SQUARE ROOT (AGL, WGL4)
	  /onequarter           % VULGAR FRACTION ONE QUARTER (AGL, WGL4)
	  /onehalf              % VULGAR FRACTION ONE HALF (AGL, WGL4)
	  /threequarters        % VULGAR FRACTION THREE QUARTERS (AGL, WGL4)
	  /Euro                 % EURO SIGN (AGL)
	% 0xC0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xD0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /multiply             % MULTIPLICATION SIGN (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xE0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xF0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /divide               % DIVISION SIGN (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	] def

	% eof
	EOF

  info "Creating encoding vector TS1-SC.enc"
  cat <<-EOF >TS1-SC.enc
	% TS1-SC.enc
	% Created by $fullname $version on $date
	% This file should be put in a tex inputs directory
	%
	% Note: This encoding vector was installed by $fullname
	%       $version.
	%
	%       Future versions of $fullname may ship with an
	%       updated version of this file.
	%
	%       This file is still more or less experimental.
	%
	% Derived from TS1-AGL.enc by Philipp Lehman <lehman@gmx.net>
	%
	% Version 0.1  2002-05-01
	%
	% Glyph names marked with "?" are uncertain; any improvements welcome

	/TS1Encoding [          % now 256 chars follow
	% 0x00
	  /Grave                % CAPITAL GRAVE ACCENT (AGL)
	  /Acute                % CAPITAL ACUTE ACCENT (AGL)
	  /.notdef              % CAPITAL CIRCUMFLEX ACCENT
	  /.notdef              % CAPITAL TILDE ACCENT
	  /Dieresis             % CAPITAL DIAERESIS (AGL)
	  /Hungarumlaut         % CAPITAL DOUBLE ACUTE ACCENT (AGL)
	  /.notdef              % CAPITAL RING ACCENT
	  /Caron                % CAPITAL CARON (AGL)
	  /.notdef              % CAPITAL BREVE ACCENT
	  /Macron               % CAPITAL MACRON (AGL)
	  /.notdef              % CAPITAL DOT ABOVE ACCENT
	  /cedilla              % CAPITAL CEDILLA (AGL, WGL4)
	  /ogonek               % CAPITAL OGONEK (AGL, WGL4)
	  /quotesinglbase       % SINGLE LOW-9 QUOTATION MARK (AGL, WGL4)
	  /.notdef
	  /.notdef
	% 0x10
	  /.notdef
	  /.notdef
	  /quotedblbase         % DOUBLE LOW-9 QUOTATION MARK (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef              % 2/3 em dash
	  /threequartersemdash  % THREE QUARTERS EM DASH (AGL)
	  /.notdef              % capital compound word mark
	  /arrowleft            % LEFTWARDS ARROW (AGL, WGL4)
	  /arrowright           % RIGHTWARDS ARROW (AGL, WGL4)
	  /uni2040              % ? TIE ACCENT (AGL)
	  /.notdef              % ? CAPITAL TIE ACCENT
	  /.notdef              % ? TIE ACCENT (variant)
	  /.notdef              % ? CAPITAL TIE ACCENT (variant)
	  /.notdef
	  /.notdef              % lowercase ascender compound word mark
	% 0x20
	  /uni0180              % LATIN SMALL LETTER B WITH STROKE (AGL)
	  /.notdef
	  /.notdef
	  /.notdef
	  /dollaroldstyle	% OLDSTYLE DOLLAR SIGN (AGL)
	  /.notdef
	  /.notdef
	  /quotesingle          % APOSTROPHE (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /asterisk             % ASTERISK (AGL, WGL4)
	  /.notdef
	  /comma                % COMMA (AGL, WGL4)
	  /.notdef              % HYPHEN (glyph variant with two strokes)
	  /period               % FULL STOP (AGL, WGL4)
	  /fraction             % ? FRACTION SLASH (AGL, WGL4) or "/slash"
	% 0x30
	  /zerooldstyle         % OLDSTYLE DIGIT ZERO (AGL)
	  /oneoldstyle          % OLDSTYLE DIGIT ONE (AGL)
	  /twooldstyle          % OLDSTYLE DIGIT TWO (AGL)
	  /threeoldstyle        % OLDSTYLE DIGIT THREE (AGL)
	  /fouroldstyle         % OLDSTYLE DIGIT FOUR (AGL)
	  /fiveoldstyle         % OLDSTYLE DIGIT FIVE (AGL)
	  /sixoldstyle          % OLDSTYLE DIGIT SIX (AGL)
	  /sevenoldstyle        % OLDSTYLE DIGIT SEVEN (AGL)
	  /eightoldstyle        % OLDSTYLE DIGIT EIGHT (AGL)
	  /nineoldstyle         % OLDSTYLE DIGIT NINE (AGL)
	  /.notdef
	  /.notdef
	  /angleleft            % LEFT-POINTING ANGLE BRACKET (AGL)
	  /minus                % MINUS SIGN (AGL, WGL4)
	  /angleright           % RIGHT-POINTING ANGLE BRACKET (AGL)
	  /.notdef
	% 0x40
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /uni2127              % INVERTED OHM SIGN (AGL)
	  /.notdef
	  /circle               % WHITE CIRCLE (AGL, WGL4)
	% 0x50
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /Omega                % OHM SIGN (AGL, WGL4 -> "/Ohm")
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef              % OPENING DOUBLE SQUARE BRACKET
	  /.notdef
	  /.notdef              % CLOSING DOUBLE SQUARE BRACKET
	  /arrowup              % UPWARDS ARROW (AGL, WGL4)
	  /arrowdown            % DOWNWARDS ARROW (AGL, WGL4)
	% 0x60
	  /grave                % GRAVE ACCENT (AGL, WGL4)
	  /.notdef
	  /.notdef              % BORN
	  /.notdef              % DIVORCED
	  /.notdef              % DIED
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef              % LEAF
	  /.notdef              % MARRIED
	  /musicalnote          % EIGTH NOTE (AGL, WGL4)
	  /.notdef
	% 0x70
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /asciitilde           % TILDE (AGL, WGL4)
	  /.notdef              % HYPHEN (glyph variant with two strokes)
	% 0x80
	  /breve                % BREVE (AGL, WGL4)
	  /caron                % CARON (AGL, WGL4)
	  /hungarumlaut         % DOUBLE ACUTE ACCENT (AGL, WGL4)
	  /dblGrave             % CAPITAL GRAVE DOUBLE ACCENT (AGL)
	  /dagger               % DAGGER (AGL, WGL4)
	  /daggerdbl            % DOUBLE DAGGER (AGL, WGL4)
	  /.notdef              % double vertical bar
	  /perthousand          % PER MILLE SIGN (AGL, WGL4)
	  /bullet               % BULLET (AGL, WGL4)
	  /uni2103              % DEGREE CELSIUS (AGL)
	  /dollaroldstyle       % OLDSTYLE DOLLAR SIGN (AGL)
	  /centoldstyle         % OLDSTYLE CENT SIGN (AGL)
	  /florin               % LATIN SMALL LETTER F WITH HOOK (AGL, WGL4)
	  /colonmonetary        % COLON SIGN (AGL)
	  /uni20A9              % WON SIGN (AGL)
	  /uni20A6              % NAIRA SIGN (AGL)
	% 0x90
	  /.notdef              % ? LATIN CAPITAL LETTER G WITH VERTICAL STROKE (guarani)
	  /.notdef              % ? LATIN CAPITAL LETTER P WITH STROKE (peso)
	  /lira                 % LIRA SIGN (AGL, WGL4 -> "/afii08941")
	  /prescription         % ? RECIPE (AGL)
	  /uni203D              % INTERROBANG (AGL)
	  /.notdef              % INTERROBANG INVERTED
	  /dong                 % DONG SIGN (AGL)
	  /trademark            % TRADE MARK SIGN (AGL, WGL4)
	  /uni2031              % PER TEN THOUSAND SIGN (AGL)
	  /.notdef              % ? PILCROW SIGN (glyph variant)
	  /uni0E3F              % THAI CURRENCY SYMBOL BAHT (AGL)
	  /afii61352            % NUMERO SIGN (AGL, WGL4)
	  /.notdef              % PERCENT SIGN (glyph variant)
	  /estimated            % ESTIMATED SYMBOL (AGL, WGL4)
	  /openbullet           % WHITE BULLET (AGL, WGL4)
	  /uni2120              % SERVICE MARK (AGL)
	% 0xA0
	  /uni2045              % LEFT SQUARE BRACKET WITH SQUILL (AGL)
	  /uni2046              % RIGHT SQUARE BRACKET WITH SQUILL (AGL)
	  /cent                 % CENT SIGN (AGL, WGL4)
	  /sterling             % POUND SIGN (AGL, WGL4)
	  /currency             % CURRENCY SIGN (AGL, WGL4)
	  /yen                  % YEN SIGN (AGL, WGL4)
	  /brokenbar            % BROKEN BAR (AGL, WGL4)
	  /section              % SECTION SIGN (AGL, WGL4)
	  /dieresis             % DIAERESIS (AGL, WGL4)
	  /copyright            % COPYRIGHT SIGN (AGL, WGL4)
	  /ordfeminine          % FEMININE ORDINAL INDICATOR (AGL, WGL4)
	  /.notdef              % COPYLEFT SIGN
	  /logicalnot           % NOT SIGN (AGL, WGL4)
	  /uni2117              % SOUND RECORDING COPYRIGHT (AGL)
	  /registered           % REGISTERED SIGN (AGL, WGL4)
	  /macron               % MACRON (AGL, WGL4)
	% 0xB0
	  /degree               % DEGREE SIGN (AGL, WGL4)
	  /plusminus            % PLUS-MINUS SIGN (AGL, WGL4)
	  /twosuperior          % SUPERSCRIPT TWO (AGL, WGL4)
	  /threesuperior        % SUPERSCRIPT THREE (AGL, WGL4)
	  /acute                % ACUTE ACCENT (AGL, WGL4)
	  /mu                   % MICRO SIGN (AGL, WGL4 -> "/mu1")
	  /paragraph            % PILCROW SIGN (AGL, WGL4)
	  /periodcentered       % MIDDLE DOT (AGL, WGL4 -> "/middot")
	  /uni203B              % REFERENCE MARK
	  /onesuperior          % SUPERSCRIPT ONE (AGL, WGL4)
	  /ordmasculine         % MASCULINE ORDINAL INDICATOR (AGL, WGL4)
	  /radical              % SQUARE ROOT (AGL, WGL4)
	  /onequarter           % VULGAR FRACTION ONE QUARTER (AGL, WGL4)
	  /onehalf              % VULGAR FRACTION ONE HALF (AGL, WGL4)
	  /threequarters        % VULGAR FRACTION THREE QUARTERS (AGL, WGL4)
	  /Euro                 % EURO SIGN (AGL)
	% 0xC0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xD0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /multiply             % MULTIPLICATION SIGN (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xE0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	% 0xF0
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /divide               % DIVISION SIGN (AGL, WGL4)
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	  /.notdef
	] def

	% eof
	EOF
}

# Misc functions used by create_fd below

create_fd_addfont() {

  local fdfile="$1"
  local texenc="$2"
  local family="$3"
  local fileenc="$4"
  local texweight="$5"
  local fileweight="$6"

  echo >>$fdfile
  info "Declaring $texenc/$family/$texweight/n"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{n} {<-> ${basefamily}${fileweight}${fileenc}}{}" >>$fdfile
  info "Declaring $texenc/$family/$texweight/sc"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sc}{<-> ${basefamily}${fileweight}c${fileenc}}{}" >>$fdfile
  info "Declaring $texenc/$family/$texweight/sl"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sl}{<-> ${basefamily}${fileweight}o${fileenc}}{}" >>$fdfile

  if [ -f "${basefamily}${fileweight}i${rawenc_wgl}.ttf" ]; then
    info "Declaring ${texenc}/$family/$texweight/it"
    echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{it}{<-> ${basefamily}${fileweight}i${fileenc}}{}" >>$fdfile
  else
    info "Adding noisy substitution ${texenc}/$family/$texweight/it --> ${texenc}/$family/$texweight/sl"
    echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{it}{<-> sub * ${family}/${texweight}/sl}{}" >>$fdfile
  fi
}

create_fd_substitute_silent() {
  trap error_internal ERR

  local fdfile="$1"
  local texenc="$2"
  local family="$3"
  local fileenc="$4"
  local texweight="$5"
  local substitute="$6"

  echo >>$fdfile

  info "Adding silent substitution ${texenc}/${family}/${texweight}/n --> ${texenc}/${family}/${substitute}/n"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{n} {<-> ssub * ${family}/${substitute}/n}{}" >>$fdfile
  info "Adding silent substitution ${texenc}/${family}/${texweight}/sc --> ${texenc}/${family}/${substitute}/sc"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sc}{<-> ssub * ${family}/${substitute}/sc}{}" >>$fdfile
  info "Adding silent substitution ${texenc}/${family}/${texweight}/sl --> ${texenc}/${family}/${substitute}/sl"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sl}{<-> ssub * ${family}/${substitute}/sl}{}" >>$fdfile
  info "Adding silent substitution ${texenc}/${family}/${texweight}/it --> ${texenc}/${family}/${substitute}/it"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{it}{<-> ssub * ${family}/${substitute}/it}{}" >>$fdfile
}

create_fd_substitute_noisy() {
  trap error_internal ERR

  local fdfile="$1"
  local texenc="$2"
  local family="$3"
  local fileenc="$4"
  local texweight="$5"
  local substitute="$6"

  echo >>$fdfile

  info "Adding noisy substitution ${texenc}/${family}/${texweight}/n --> ${texenc}/${family}/${substitute}/n"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{n} {<-> sub * ${family}/${substitute}/n}{}" >>$fdfile
  info "Adding noisy substitution ${texenc}/${family}/${texweight}/sc --> ${texenc}/${family}/${substitute}/sc"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sc}{<-> sub * ${family}/${substitute}/sc}{}" >>$fdfile
  info "Adding noisy substitution ${texenc}/${family}/${texweight}/sl --> ${texenc}/${family}/${substitute}/sl"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{sl}{<-> sub * ${family}/${substitute}/sl}{}" >>$fdfile
  info "Adding noisy substitution ${texenc}/${family}/${texweight}/it --> ${texenc}/${family}/${substitute}/it"
  echo "\\DeclareFontShape{$texenc}{$family}{$texweight}{it}{<-> sub * ${family}/${substitute}/it}{}" >>$fdfile
}

# Create font declaration files

create_fd() {
  trap error_internal ERR

  local fdfile="$1"
  local texenc="$2"
  local family="$3"
  local fileenc="$4"

  # headers

  info "Creating $fdfile"

  echo -n >$fdfile
  echo -e "% ${fdfile}\n% Created by $fullname $version on $date" >>$fdfile
  echo -e "% This file should be put in a tex inputs directory\n" >>$fdfile
  echo "\\ProvidesFile{${fdfile}}[$texdate ${texenc}/${family} font definitions]" >>$fdfile
  echo >>$fdfile
  echo "\\DeclareFontFamily{$texenc}{$family}{$family_init}" >>$fdfile

  # ultra light

  if [ -f "${basefamily}a${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "ul" "a"
  fi

  # extra light

  if [ -f "${basefamily}j${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "el" "j"
  fi

  # light

  if [ -f "${basefamily}l${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "l" "l"
  fi

  # book

  if [ -f "${basefamily}k${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "k" "k"
  fi
  
  # regular

  if [ -f "${basefamily}r${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "m" "r"
  else
    if [ "$mffallback" = "k" ]; then
      create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "m" "$mffallback"
    else
      create_fd_substitute_noisy "$fdfile" "$texenc" "$family" "$fileenc" "m" "$mffallback"
    fi
  fi

  # medium

  if [ -f "${basefamily}m${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "mb" "m"
  fi

  # demibold

  if [ -f "${basefamily}d${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "db" "d"
  fi

  # semibold

  if [ -f "${basefamily}s${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "sb" "s"
  fi

  # bold

  if [ -f "${basefamily}b${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "b" "b"
  else
    if [ "$bffallback" = "mb" -o "$bffallback" = "sb" -o "$bffallback" = "db" ]; then
      create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "b" "$bffallback"
    else
      create_fd_substitute_noisy "$fdfile" "$texenc" "$family" "$fileenc" "b" "$bffallback"
    fi
  fi

  # extra bold

  if [ -f "${basefamily}x${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "eb" "x"
  fi

  # heavy

  if [ -f "${basefamily}h${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "hb" "h"
    if [ "$ebdefault" = "hb" ]; then
      create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "eb" "hb"
    fi
  fi

  # black

  if [ -f "${basefamily}c${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "cb" "c"
    if [ "$ebdefault" = "cb" ]; then
      create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "eb" "cb"
    fi
  fi

  # ultra bold

  if [ -f "${basefamily}u${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "ub" "u"
  fi

  # poster

  if [ -f "${basefamily}p${rawenc_wgl}.ttf" ]; then
    create_fd_addfont "$fdfile" "$texenc" "$family" "$fileenc" "pb" "p"
    if [ "$ubdefault" = "pb" ]; then
      create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "ub" "pb"
    fi
  fi

  # bold extended

  if [ "$bfdefault" = "mb" -o "$bfdefault" = "sb"\
    -o "$bfdefault" = "db" -o "$bfdefault" = "b" ]
  then
    create_fd_substitute_silent "$fdfile" "$texenc" "$family" "$fileenc" "bx" "$bfdefault"
  else
    create_fd_substitute_noisy "$fdfile" "$texenc" "$family" "$fileenc" "bx" "$bfdefault"
  fi

  # end input

  echo >>$fdfile
  echo "\\endinput" >>$fdfile
  echo >>$fdfile
}

# Tag map files

create_map_begin() {
  local family="$1"
  echo "% [BEG $family] -- Created by $fullname $version on $date" >>$file_map_ttf2pk
  echo "% [BEG $family] -- Created by $fullname $version on $date" >>$pdftexmap
}

create_map_end() {
  local family="$1"
  echo "% [END $family] -- Created by $fullname $version on $date" >>$file_map_ttf2pk
  echo "% [END $family] -- Created by $fullname $version on $date" >>$pdftexmap
}

# Pretty printing for map files

create_map_fill() {
  trap error_internal ERR

  local strg="$2"
  if [ ${#strg} -lt $1 ]; then
    local cnt=${#strg}
    while [ $cnt -lt $1 ]; do
      local strg="$strg "
      local cnt=$(($cnt + 1))
    done
  else
    local strg="$strg "
  fi
  echo -n "$strg"
}

# Warning: slanting not supported by pdftex

create_map_warning() {
  trap error_internal ERR

  cat <<-EOF >>$pdftexmap
	% Note: When this version of $fullname was released, pdfTeX did not support slanting of
	%       TrueType fonts. The relevant lines below are therefore commented out. See the
	%       ttf2tex manual and the pdfTeX documentation for details.
	EOF
}

# Create afm files

create_afm() {
  trap error_internal ERR

  for series in $fontseries; do
    for font in "${basefamily}${series}${rawenc_wgl}.ttf" "${basefamily}${series}i${rawenc_wgl}.ttf"; do
      if [ -e $font ]; then
	base="${font%${rawenc_wgl}.ttf}"
	info "Running ttf2afm on $font"
	ttf2afm -a -o ${base}${rawenc_wgl}.afm ./$font 2>ttf2afm.log
	while read output; do info_external "ttf2afm" "$output"; done <ttf2afm.log
      fi
    done
  done
}

# Create metrics and map files (upright)

create_tfm_upright() {
  trap error_internal ERR

  local flags="$1"
  local vector="$2"
  local enc="$3"
  local rawenc="$4"

  for series in $fontseries; do
    font="${basefamily}${series}${rawenc_wgl}.ttf"
    if [ -e $font ]; then
      base="${font%${rawenc_wgl}.ttf}"

      info "Running ttf2tfm on $font [vector: ${vector}]"
      ttf2tfm ./$font $flags -T $vector -v ${base}${enc}.vpl ${base}${rawenc}.tfm 2>ttf2tfm.log | while read texfontname ttfontname encoding; do
	create_map_fill 16 "$texfontname" >>$file_map_ttf2pk
	create_map_fill 24 "$ttfontname" >>$file_map_ttf2pk
	echo "$encoding" >>$file_map_ttf2pk
      done
      while read output; do info_external "ttf2tfm" "$output"; done <ttf2tfm.log

      psname="$(grep "^FontName " ${base}${rawenc_wgl}.afm | cut -d ' ' -f 2-)"

      info "Adding $font to $pdftexmap [vector: ${vector}]"
      create_map_fill 16 "${base}${rawenc}" >>$pdftexmap
      create_map_fill 56 "$psname" >>$pdftexmap
      echo "<$vector <$font" >>$pdftexmap

    fi
  done
}

# Create metrics and map files (italic)

create_tfm_italic() {
  trap error_internal ERR

  local flags="$1"
  local vector="$2"
  local enc="$3"
  local rawenc="$4"

  for series in $fontseries; do
    font="${basefamily}${series}i${rawenc_wgl}.ttf"
    if [ -e $font ]; then
      base="${font%${rawenc_wgl}.ttf}"

      info "Running ttf2tfm on $font [vector: ${vector}]"
      ttf2tfm ./$font $flags -T $vector -v ${base}${enc}.vpl ${base}${rawenc}.tfm 2>ttf2tfm.log | while read texfontname ttfontname encoding; do
	create_map_fill 16 "$texfontname" >>$file_map_ttf2pk
	create_map_fill 24 "$ttfontname" >>$file_map_ttf2pk
	echo "$encoding" >>$file_map_ttf2pk
      done
      while read output; do info_external "ttf2tfm" "$output"; done <ttf2tfm.log

      psname="$(grep "^FontName " ${base}${rawenc_wgl}.afm | cut -d ' ' -f 2-)"

      info "Adding $font to $pdftexmap [vector: ${vector}]"
      create_map_fill 16 "${base}${rawenc}" >>$pdftexmap
      create_map_fill 56 "$psname" >>$pdftexmap
      echo "<$vector <$font" >>$pdftexmap

    fi
  done
}

# Create metrics and map files (real small caps)

create_tfm_smallcaps_real() {
  trap error_internal ERR

  local flags="$1"
  local vector="$2"
  local enc="$3"
  local rawenc="$4"

  for series in $fontseries; do
    font="${basefamily}${series}${rawenc_wgl}.ttf"
    if [ -e $font ]; then
      base="${font%${rawenc_wgl}.ttf}"

      info "Running ttf2tfm on $font [vector: ${vector}]"
      ttf2tfm ./$font $flags -T $vector -v ${base}c${enc}.vpl ${base}c${rawenc}.tfm 2>ttf2tfm.log | while read texfontname ttfontname encoding; do
	create_map_fill 16 "$texfontname" >>$file_map_ttf2pk
	create_map_fill 24 "$ttfontname" >>$file_map_ttf2pk
	echo "$encoding" >>$file_map_ttf2pk
      done
      while read output; do info_external "ttf2tfm" "$output"; done <ttf2tfm.log

      psname="$(grep '^FontName' ${base}${rawenc_wgl}.afm | cut -d ' ' -f 2-)"

      info "Adding $font to $pdftexmap [vector: ${vector}]"
      create_map_fill 16 "${base}c${rawenc}" >>$pdftexmap
      create_map_fill 56 "$psname" >>$pdftexmap
      echo "<$vector <$font" >>$pdftexmap

    fi
  done
}

# Create metrics and map files (faked small caps)

create_tfm_smallcaps_fake() {
  trap error_internal ERR

  local flags="$1"
  local vector="$2"
  local enc="$3"
  local rawenc="$4"

  for series in $fontseries; do
    font="${basefamily}${series}${rawenc_wgl}.ttf"
    if [ -e $font ]; then
      base="${font%${rawenc_wgl}.ttf}"

      info "Running ttf2tfm on $font [vector: ${vector}] [small caps height: $capsheight]"
      ttf2tfm ./$font $flags -T $vector -c $capsheight -V ${base}c${enc}.vpl ${base}${rawenc}.tfm >/dev/null 2>ttf2tfm.log
      while read output; do info_external "ttf2tfm" "$output"; done <ttf2tfm.log

    fi
  done
}

# Create metrics and map files (slanted)

create_tfm_slanted() {
  trap error_internal ERR

  local flags="$1"
  local vector="$2"
  local enc="$3"
  local rawenc="$4"

  for series in $fontseries; do
    font="${basefamily}${series}${rawenc_wgl}.ttf"
    if [ -e $font ]; then
      base="${font%${rawenc_wgl}.ttf}"

      info "Running ttf2tfm on $font [vector: ${vector}] [slant factor: $slantfactor]"
      ttf2tfm ./$font $flags -T $vector -s $slantfactor -v ${base}o${enc}.vpl ${base}o${rawenc}.tfm 2>ttf2tfm.log | while read texfontname ttfontname encoding; do
	create_map_fill 16 "$texfontname" >>$file_map_ttf2pk
	create_map_fill 24 "$ttfontname" >>$file_map_ttf2pk
	echo "$encoding" >>$file_map_ttf2pk
      done
      while read output; do info_external "ttf2tfm" "$output"; done <ttf2tfm.log

      psname="$(grep '^FontName' ${base}${rawenc_wgl}.afm | cut -d ' ' -f 2-)"

      # slanting not supported by pdftex

      info "Adding $font to $pdftexmap [vector: ${vector}] [slant factor: $slantfactor] *disabled*"
      create_map_fill 16 "% ${base}o${rawenc}" >>$pdftexmap
      create_map_fill 32 "$psname" >>$pdftexmap
      create_map_fill 24 "\"$slantfactor SlantFont\"" >>$pdftexmap
      echo "<$vector <$font" >>$pdftexmap

    fi
  done
}

# Create virtual fonts

create_vf() {
  trap error_internal ERR

  for font in *.vpl; do
     base=$(basename $font .vpl)
     info "Running vptovf on $font"
     vptovf ./$font ${base}.vf ${base}.tfm | while read output; do info_external "vptovf" "$output"; done
  done
}

# Copy/move files safely, used by install_files below

install_files_copy() {
  trap error_internal ERR

  if [ -e "$2/$1" ]; then
    if [ $overwrite = 1 ]; then
      info "Overwriting $1"
      cp $1 $2
      chmod 0644 "$2/$1"
    else
      info "File $2/$1 exists"
      info "Skipping $1"
    fi
  else
    info "Installing $1"
    cp $1 $2
    chmod 0644 "$2/$1"
  fi
}

install_files_move() {
  trap error_internal ERR

  if [ -e "$2/$1" ]; then
    if [ $overwrite = 1 ]; then
      info "Overwriting $1"
      mv $1 $2
      chmod 0644 "$2/$1"
    else
      info "File $2/$1 exists"
      info "Skipping $1"
    fi
  else
    info "Installing $1"
    mv $1 $2
    chmod 0644 "$2/$1"
  fi
}

# Sort files in working directory

sort_files() {
  trap error_internal ERR

  for dir in $path_afm $path_tfm $path_ttf $path_vf $path_fd; do
    mkdir -m 0755 -p ${dir}/${foundry}/${fontname}
  done
  for dir in $path_map_ttf2pk $path_enc_ttf2pk $path_map_pdftex $path_enc_pdftex; do
    mkdir -m 0755 -p $dir
  done
  for file in ${basefamily}*.afm; do
    info "Moving $file"
    mv $file ${path_afm}/${foundry}/${fontname}
  done
  for file in ${basefamily}*.tfm; do
    info "Moving $file"
    mv $file ${path_tfm}/${foundry}/${fontname}
  done
  for file in ${basefamily}*.ttf; do
    info "Copying $file"
    cp $file ${path_ttf}/${foundry}/${fontname}
  done
  for file in ${basefamily}*.vf; do
    info "Moving $file"
    mv $file ${path_vf}/${foundry}/${fontname}
  done
  for file in *${basefamily}*.fd; do
    info "Moving $file"
    mv $file ${path_fd}/${foundry}/${fontname}
  done
  for file in *.enc; do
    info "Copying $file"
    cp $file ${path_enc_ttf2pk}
    cp $file ${path_enc_pdftex}
  done
  info "Moving $pdftexmap"
  mv $pdftexmap ${path_map_pdftex}/${pdftexmap}
  info "Updating $file_map_ttf2pk"
  cat $file_map_ttf2pk >>${path_map_ttf2pk}/${file_map_ttf2pk}
}

# Install files into tex tree

install_files() {
  trap error_internal ERR

  for dir in $path_afm $path_tfm $path_ttf $path_vf $path_fd; do
    mkdir -m 0755 -p ${texmfinstall}/${dir}/${foundry}/${fontname}
  done
  for dir in $path_map_ttf2pk $path_enc_ttf2pk $path_map_pdftex $path_enc_pdftex; do
    mkdir -m 0755 -p ${texmfinstall}/${dir}
  done
  for file in ${basefamily}*.afm; do
    install_files_move $file "${texmfinstall}/${path_afm}/${foundry}/${fontname}"
  done
  for file in ${basefamily}*.tfm; do
    install_files_move $file "${texmfinstall}/${path_tfm}/${foundry}/${fontname}"
  done
  for file in ${basefamily}*.ttf; do
    install_files_copy $file "${texmfinstall}/${path_ttf}/${foundry}/${fontname}"
  done
  for file in ${basefamily}*.vf; do
    install_files_move $file "${texmfinstall}/${path_vf}/${foundry}/${fontname}"
  done
  for file in *${basefamily}*.fd; do
    install_files_move $file "${texmfinstall}/${path_fd}/${foundry}/${fontname}"
  done
  for file in *.enc; do
    install_files_copy $file "${texmfinstall}/${path_enc_ttf2pk}"
    install_files_copy $file "${texmfinstall}/${path_enc_pdftex}"
  done
  cat $file_map_ttf2pk >>${texmfinstall}/${path_map_ttf2pk}/${file_map_ttf2pk}
  install_files_move $pdftexmap ${texmfinstall}/${path_map_pdftex}

  # update file database
  
  info "Running mktexlsr to update file database"
  mktexlsr $texmfinstall >mktexlsr.log 2>&1
  while read output; do info_external "mktexlsr" "$output"; done <mktexlsr.log
}

# Trap internal errors

trap error_internal ERR

# Read config file

for confpath in $confpath; do
  if [ -f "${confpath}${fullname}.cfg" ]; then
    conffile="${confpath}${fullname}.cfg"
    break
  fi
done

if [ -n "${conffile}" ]; then
  source "${conffile}"
else
  error "Failed to find configuration file ${fullname}.cfg!"
fi

# Set defaults

path_afm="${path_afm:-fonts/afm}"
path_tfm="${path_tfm:-fonts/tfm}"
path_vf="${path_vf:-fonts/vf}"
path_ttf="${path_ttf:-fonts/truetype}"
path_fd="${path_fd:-tex/latex/ttfonts}"
path_map_pdftex="${path_map_pdftex:-pdftex/config}"
path_enc_pdftex="${path_enc_pdftex:-pdftex/config}"
path_map_ttf2pk="${path_map_ttf2pk:-ttf2pk}"
path_enc_ttf2pk="${path_enc_ttf2pk:-ttf2pk}"
file_map_ttf2pk="${file_map_ttf2pk:-ttfonts.map}"

slantfactor="0.167"
capsheight="0.800"
confirm=1
inst=0
user=0
expert=0
overwrite=0
flags_common="-q"
texenc_t1="8t"
texenc_ts1="8c"
texenc_t1x="9e"
texenc_ts1x="9c"
texenc_t1j="9d"
rawenc_wgl="16"
rawenc_t1="16t"
rawenc_ts1="16c"
rawenc_t1x="17e"
rawenc_ts1x="17c"
rawenc_t1j="17d"

# Parse command line

if [ $# -lt 1 ]; then error_usage; fi

while [ -n "$1" ]; do
  case $1 in
    --dump-vectors)
		dump_vectors
        	exit 0
		;;
    -h|--help)	usage
    		exit 0
		;;
    -b|--batch)	confirm=0
        	shift
		;;
    -l|--log)   transcript=1
        	shift
		;;
    -i|--install)
    		inst=1
        	shift
		;;
    -u|--user)	user=1
		inst=1
        	shift
		;;
    -t|--typewriter)
		family_init="\\hyphenchar\\font=-1"
        	shift
		;;
    -x|--expert)
		expert=1
        	shift
		;;
    -n|--ps-names)
		flags_psnames="-n"
        	shift
		;;
    -N|--ps-names-only)
		flags_psnames="-N"
        	shift
		;;
    --overwrite)
		overwrite=1
        	shift
		;;
    -f|--foundry) 
		if [ -z "$2" -o -z "${2%%-*}" ]; then error_usage; fi
    		foundry="${2// /}"
        	shift
        	shift
		;;
    -o|--font)	if [ -z "$2" -o -z "${2%%-*}" ]; then error_usage; fi
    		fontname="${2// /}"
        	shift
        	shift
		;;
    -c|--caps)	if [ -z "$2" -o -z "${2%%-*}" ]; then error_usage; fi
    		capsheight="$2"
        	shift
        	shift
		;;
    -s|--slant)	if [ -z "$2" -o -z "${2%%-*}" ]; then error_usage; fi
    		slantfactor="$2"
        	shift
        	shift
		;;
    -*)		error_usage
		;;
    [A-Za-z0-9][A-Za-z0-9][A-Za-z0-9]*)
    		basefamily="$1"
        	shift
		;;
    *)		error_usage
		;;
  esac
done

[ -n "$basefamily" -a -n "$foundry" -a -n "$fontname" ] || error_usage
pdftexmap="${basefamily}.map"

if [ $inst = 1 ]; then
  if [ $user = 1 ]; then
    texmfinstall="$(kpsexpand '$HOMETEXMF')"
  else
    texmfinstall="$(kpsexpand '$TEXMFLOCAL')"
  fi
  [ -n "$texmfinstall" ] || error "Failed to get \$TEXMFLOCAL from kpathsea!"
  [ -d "$texmfinstall" ] || error "Directory $texmfinstall does not exist!"
  [ -w "$texmfinstall" ] || error "Directory $texmfinstall is not writeable!"
  if [ $overwrite = 1 ]; then
    echo
    info "Running in overwrite mode!"
    echo
  fi
fi

# Traps, umask

trap clean EXIT
trap "error 'Received SIGINT!'" SIGINT
umask 022

# Let's go

if [ $transcript = 1 ]; then
  echo -e "\nTranscript of $fullname session on $date\n" >$logfile
fi

check_requirements
info_paths
prompt_confirm
check_available

info_map_available "$basefamily"

if [ $expert = 1 ]; then
  info_map_available "${basefamily}x"
  info_map_available "${basefamily}j"
fi

prompt_confirm
dump_vectors
create_afm
create_map_begin "$basefamily"

if [ $expert = 1 ]; then
  # standard
  create_tfm_upright "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_upright "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_italic "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_italic "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_smallcaps_real "$flags_common -N" "T1-SC.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_smallcaps_real "$flags_common -N" "TS1-SC.enc" "$texenc_ts1" "$rawenc_ts1"

  create_map_warning
  create_tfm_slanted "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_slanted "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  # expert
  create_tfm_upright "$flags_common -N" "T1-WGL4.enc" "$texenc_t1x" "$rawenc_t1x"
  create_tfm_upright "$flags_common -N" "TS1-AGL.enc" "$texenc_ts1x" "$rawenc_ts1x"

  create_tfm_italic "$flags_common -N" "T1-WGL4.enc" "$texenc_t1x" "$rawenc_t1x"
  create_tfm_italic "$flags_common -N" "TS1-AGL.enc" "$texenc_ts1x" "$rawenc_ts1x"

  create_tfm_smallcaps_real "$flags_common -N" "T1-SC.enc" "$texenc_t1x" "$rawenc_t1x"
  create_tfm_smallcaps_real "$flags_common -N" "TS1-SC.enc" "$texenc_ts1x" "$rawenc_ts1x"

  create_map_warning
  create_tfm_slanted "$flags_common -N" "T1-WGL4.enc" "$texenc_t1x" "$rawenc_t1x"
  create_tfm_slanted "$flags_common -N" "TS1-AGL.enc" "$texenc_ts1x" "$rawenc_ts1x"

  # osf
  create_tfm_upright "$flags_common -N" "T1-OSF.enc" "$texenc_t1j" "$rawenc_t1j"
  create_tfm_italic "$flags_common -N" "T1-OSF.enc" "$texenc_t1j" "$rawenc_t1j"
  create_tfm_smallcaps_real "$flags_common -N" "T1-SCOSF.enc" "$texenc_t1j" "$rawenc_t1j"

  create_map_warning
  create_tfm_slanted "$flags_common -N" "T1-OSF.enc" "$texenc_t1j" "$rawenc_t1j"

elif [ $expert = 0 -a -n "$flags_psnames" ]; then
  create_tfm_upright "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_upright "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_italic "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_italic "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_smallcaps_fake "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_smallcaps_fake "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_map_warning
  create_tfm_slanted "$flags_common $flags_psnames" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  create_tfm_slanted "$flags_common $flags_psnames" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

else
  create_tfm_upright "$flags_common" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  # Hack required to get the euro currency symbol into TS1 (for ttf2pk)
  create_tfm_upright "$flags_common -r .c0x20AC Euro" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_italic "$flags_common" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  # Hack required to get the euro currency symbol into TS1 (for ttf2pk)
  create_tfm_italic "$flags_common -r .c0x20AC Euro" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_tfm_smallcaps_fake "$flags_common" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  # Hack required to get the euro currency symbol into TS1 (for ttf2pk)
  create_tfm_smallcaps_fake "$flags_common -r .c0x20AC Euro" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"

  create_map_warning
  create_tfm_slanted "$flags_common" "T1-WGL4.enc" "$texenc_t1" "$rawenc_t1"
  # Hack required to get the euro currency symbol into TS1 (for ttf2pk)
  create_tfm_slanted "$flags_common -r .c0x20AC Euro" "TS1-AGL.enc" "$texenc_ts1" "$rawenc_ts1"
fi

create_map_end "$basefamily"
create_vf

create_fd "t1${basefamily}.fd" "T1" "$basefamily" "$texenc_t1"
create_fd "ts1${basefamily}.fd" "TS1" "$basefamily" "$texenc_ts1"

if [ $expert = 1 ]; then
  create_fd "t1${basefamily}x.fd" "T1" "${basefamily}x" "$texenc_t1x"
  create_fd "ts1${basefamily}x.fd" "TS1" "${basefamily}x" "$texenc_ts1x"

  create_fd "t1${basefamily}j.fd" "T1" "${basefamily}j" "$texenc_t1j"
  create_fd "ts1${basefamily}j.fd" "TS1" "${basefamily}j" "$texenc_ts1x"
fi

if [ $inst = 1 ]; then
  install_files
else
  sort_files
fi

info "Finished"

# EOF
