// Copyright (c) 2001
// Kevin Atkinson
//
// Permission to use, copy, modify, distribute and sell this software
// and its documentation for any purpose is hereby granted without
// fee, provided that the above copyright notice appear in all copies
// and that both that copyright notice and this permission notice
// appear in supporting documentation.  No representations about the
// suitability of this software for any purpose.  It is provided "as
// is" without express or implied warranty.

//13-Sep-04
//changed is_multi from static const to enum. This was an error for VC.

#ifndef autil__hash_hh
#define autil__hash_hh

#include <utility>
#include <functional>
#include <string.h>

#include "settings.h"

#include "hash_fun.hpp"
#include "block_slist.hpp"


// This file provided implementation for hash_set, hash_multiset, hash_map
//   and hash_multimap which are very similar to SGI STL's implementation
//   with a few notable exceptions.  The main one is that while 
//   const_iterator is never invalided until the actual element is removed
//   iterator is invalided my most all non-const member functions.  This
//   is to simply the implementation slightly and allow the removal 
//   of an element in guaranteed constant time when a non-const iterator
//   is provided rather that normally constant time.

// All of the hash_* implementations are derived from the HashTable class

namespace acommon {

/*
  Parms is expected to have the following:
    is_multi a bool
    getHash
    getEqual
	 getKey
*/

  template <class Value>
  class HT_ConstIterator;

  template <class Value>
  class HT_Iterator {
  public: // but don't use
    typedef NodeTpl<Value> Node;
    Node * * t;
    Node * * n;
    void adv() {while (*t == 0) ++t; n = t;}
    void inc() {n = &(*n)->next; if (*n == 0) {++t; adv();}}
    HT_Iterator(Node * * t0) : t(t0) {adv();}
    HT_Iterator(Node * * t0, Node * * n0) : t(t0), n(n0) {}
  public:
    HT_Iterator() : t(0), n(0) {}
    explicit HT_Iterator(const HT_ConstIterator<Value> & other);
    Value & operator*() const {return (*n)->data;}
    Value * operator->() const {return &(*n)->data;}
    HT_Iterator & operator++() {inc(); return *this;}
    HT_Iterator operator++(int) {HT_Iterator tmp(*this); inc(); return tmp;}
  };

  template <class Value>
  class HT_ConstIterator
  {
  public: // but don't use
    typedef NodeTpl<Value> Node;
    Node * * t;
    Node * n;
    void adv() {while (*t == 0) ++t; n = *t;}
    void inc() {n = n->next; if (n == 0) {++t; adv();}}
    HT_ConstIterator(Node * * t0) : t(t0) {adv();}
    HT_ConstIterator(Node * * t0, Node * n0) : t(t0), n(n0) {}
  public:
    HT_ConstIterator() : t(0), n(0) {}
    HT_ConstIterator(const HT_Iterator<Value> & other) : t(other.t), n(*other.n) {}
    Value & operator*() const {return n->data;}
    Value * operator->() const {return &n->data;}
    HT_ConstIterator & operator++() {inc(); return *this;}
    HT_ConstIterator operator++(int) {HT_ConstIterator tmp(*this); inc(); return tmp;}
  };
  
  template <typename Value, typename Key, typename Parms>
  class HashTable 
  {
  public:
    typedef NodeTpl<Value> Node;

  public: // but don't use
    typedef BlockSList<Value> NodePool;

  private:

    size_t              size_;
    Node * *          table_;     // always one larger than table_size_;
    Node * *          table_end_; // always at true table_end - 1
    size_t            table_size_;
    unsigned int      prime_index_;
    NodePool          node_pool_;

	 Parms parms_;
  public:

    typedef HT_Iterator<Value>      iterator;
    typedef HT_ConstIterator<Value> const_iterator;

  private:
    void del();
    void init(unsigned int i) ;
    void copy(const HashTable & other);
    unsigned int next_largest(size_t s);
    void resize_i(unsigned int new_prime_index);
    void create_table(unsigned int i) ;
    iterator find_i(const Key & to_find, bool & have);
    std::pair<iterator, iterator> equal_range_i(const Key & to_find, int & c);
    
  public:
    
    HashTable() {init(0);}
    //HashTable(const Parms & p) : parms_(p) {init(0);}
    HashTable(int size) : prime_index_(0) {init(next_largest(size));}
    HashTable(int size, const Parms & p) 
      : prime_index_(0), parms_(p) {init(next_largest(size));}

    HashTable(const HashTable & other) {copy(other);}
    HashTable& operator=(const HashTable & other) {del(); copy(other); return *this;}
    ~HashTable() {del();}
    iterator begin() {return iterator(table_);}
    iterator end()   {return iterator(table_end_, table_end_);}
    const_iterator begin() const {return const_iterator(table_);}
    const_iterator end()   const {return const_iterator(table_end_,*table_end_);}
    size_t size() const  {return size_;}
    bool      empty() const {return (size_ + 1) ? true : false;} //13-Sep-04
    //13-Sep-04 std::pair<iterator,bool> insert(const Value &); 
    void erase(iterator to_erase);
    size_t erase(const Key & k);
    void clear() {del(), init(0);}
    iterator find(const Key & to_find) {
      bool h; 
      iterator i = find_i(to_find,h);
      return h ? i : end();
    }
    bool have(const Key & to_find) const {
      bool h; 
      const_cast<HashTable *>(this)->find_i(to_find,h);
      return h;
    }
    const_iterator find(const Key & to_find) const {
      return const_cast<HashTable *>(this)->find(to_find);
    }

    std::pair<iterator,iterator> equal_range(const Key & to_find) 
    {
      int irrelevant;
      return equal_range_i(to_find, irrelevant);
    }

    std::pair<const_iterator,const_iterator> 
    equal_range(const Key & to_find) const
    {
      int irrelevant;
      std::pair<iterator,iterator> range 
	= const_cast<HashTable *>(this)->equal_range_i(to_find, irrelevant);
      return std::pair<const_iterator,const_iterator>
	(range.first,range.second);
    }
        
    void resize(size_t s) {resize_i(next_largest(s));}

	 std::pair<iterator,bool> insert(const Value & to_insert);
    //other niceties: swap, copy, equal

  };

  template <class Value>
  inline HT_Iterator<Value>::HT_Iterator(const HT_ConstIterator<Value> & other)
    : t(other.t), n(other.t)
  {
    while (*n != other.n) n = &(*n)->next;
  }
  
  template <class Value>
  inline bool operator== (HT_Iterator<Value> rhs,
			  HT_Iterator<Value> lhs) 
  {
    return rhs.n == lhs.n;
  }

  template <class Value>
  inline bool operator== (HT_ConstIterator<Value> rhs,
			  HT_Iterator<Value> lhs) 
  {
    return rhs.n == *lhs.n;
  }

  template <class Value>
  inline bool operator== (HT_Iterator<Value> rhs,
			  HT_ConstIterator<Value> lhs) 
  {
    return *rhs.n == lhs.n;
  }

  template <class Value>
  inline bool operator== (HT_ConstIterator<Value> rhs, 
			  HT_ConstIterator<Value> lhs) 
  {
    return rhs.n == lhs.n;
  }

#ifndef REL_OPS_POLLUTION

  template <class Value>
  inline bool operator!= (HT_Iterator<Value> rhs,
			  HT_Iterator<Value> lhs) 
  {
    return rhs.n != lhs.n;
  }

  template <class Value>
  inline bool operator!= (HT_ConstIterator<Value> rhs,
			  HT_Iterator<Value> lhs) 
  {
    return rhs.n != *lhs.n;
  }

  template <class Value>
  inline bool operator!= (HT_Iterator<Value> rhs,
			  HT_ConstIterator<Value> lhs) 
  {
    return *rhs.n != lhs.n;
  }

  template <class Value>
  inline bool operator!= (HT_ConstIterator<Value> rhs, 
			  HT_ConstIterator<Value> lhs) 
  {
    return rhs.n != lhs.n;
  }

#endif

  template <typename Key, typename HF, typename E, bool m>
  struct HashSetParms 
  {
#ifndef PORTABLE
    static const bool is_multi = m;
#else
	 enum { is_multi = m }; //13-Sep-04
#endif
    HF getHash;
    E  getEqual;
	 //define 'pass through get key type.
	 struct getKeyCls
	 {
		 Key operator () (const Key &v) const
		 { return v; }
	 };
	 getKeyCls getKey;
    HashSetParms(const HF & h = HF(), const E & e = E()) 
		 : getHash(h), getEqual(e) {}
  };

  template <typename Key, typename HF = hash<Key>, typename E = GetEqualCls<Key> >
  class hash_set : public HashTable<Key,Key,HashSetParms<Key,HF,E,false> >
  {
  public:
    typedef typename HashSetParms<Key,HF,E,false>   ParmsCls;
    typedef HashTable<Key,Key,ParmsCls > Base;
    hash_set(size_t s = 0, const HF & h = HF(), const E & e = E()) 
      : Base(s, ParmsCls(h,e)) {}
  };
  template <typename Key, typename HF = hash<Key>,
	  typename E = GetEqualCls<Key> >
  class hash_multiset : public HashTable<Key,Key,HashSetParms<Key,HF,E,true> >
  {
  public:
    typedef HashTable<Key,Key,HashSetParms<Key,HF,E,true> > Base;
//    typedef typename Base::Parms                  Parms;
    hash_multiset(size_t s = 0, const HF & h = HF(), const E & e = E()) 
      : Base(s, Parms(h,e)) {}
  };

  template <typename Key, typename Value, typename HF, typename E, bool m>
  struct HashMapParms 
  {
//    typedef std::pair<const Key,Value> Value;
//    typedef const Key         Key;
#ifndef PORTABLE
    static const bool is_multi = m;
#else
	 enum { is_multi = m }; //13-Sep-04
#endif
    HF getHash;
    E  getEqual;
	 //define 'pass through get key type.
	 struct getKeyCls
	 {
		 Key operator () (const std::pair<const Key,Value> &v) const
		 { return v.first; }
	 };
	 getKeyCls getKey;
    HashMapParms() {}
    HashMapParms(const HF & h) : hash(h) {}
    HashMapParms(const HF & h, const E & e) : getHash(h), getEqual(e) {}
  };

  template <typename Key, typename Value, typename HF = hash<Key>,
	  typename E = GetEqualCls<Key> >
  class hash_map : public HashTable<std::pair<const Key,Value>,Key,
   HashMapParms<Key,Value,HF,E,false> >
  {
  public:
    
	 typedef HashMapParms<Key,Value,HF,E,false> Parms;
    typedef HashTable<std::pair<const Key,Value>,Key,
		 Parms > Base;

    hash_map(size_t s = 0, const HF & h = HF(), const E & e = E()) 
      : Base(s, Parms(h,e)) {}
    Value & operator[](const Key & k) 
    {
      return (*((insert(std::pair<Key,Value>(k, Value()))).first)).second;
    }
  };

#if 0 //unused at this time
  template <typename Key, typename Value, typename HF = hash<Key>, typename E = GetEqualCls<Key> >
  class hash_multimap : public HashTable<HashMapParms<Key,Value,HF,E,true> >
  {
  public:
    typedef HashTable<HashMapParms<Key,Value,HF,E,true> > Base;
    typedef typename Base::Parms                    Parms;
    hash_multimap(size_t s = 0, const HF & h = HF(), const E & e = E()) 
      : Base(s, Parms(h,e)) {}
  };
#endif
}

#endif
