/*
 * Created on Mar 20, 2003
 *
 * @author henkel@cs.colorado.edu
 * 
 */
package bibtex.dom.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import bibtex.dom.BibtexAbstractEntry;
import bibtex.dom.BibtexAbstractValue;
import bibtex.dom.BibtexConcatenatedValue;
import bibtex.dom.BibtexConstants;
import bibtex.dom.BibtexEntry;
import bibtex.dom.BibtexFile;
import bibtex.dom.BibtexPreamble;
import bibtex.dom.BibtexStringReference;
import bibtex.dom.BibtexString;
import bibtex.dom.BibtexStringDefinition;
import bibtex.dom.BibtexToplevelComment;

/**
 * @author henkel
 */
public final class Expansions {

	public static class ExpansionException extends Exception {
		ExpansionException(String message) {
			super(message);
		}
	}

	private static final HashSet monthAbbreviations = new HashSet();

	static {
		for (int i = 0;
			i < BibtexConstants.MONTH_ABBREVIATIONS.length;
			i++) {
			monthAbbreviations.add(BibtexConstants.MONTH_ABBREVIATIONS[i]);
		}
	}

	/** 
	 * this method walks over all entries (including strings!) and expands string references.
	 * Thus, after the execution of this function, all fields contain BibtexString or
	 * BibtexNumber entries. Exception: 1) the crossref fields 2) the standard 3-letter
	 * abbreviations for months.
	 * 
	 * @param bibtexFile
	 */
	public static void expandStringReferences(
		BibtexFile bibtexFile,
		boolean removeStringDefinitions)
		throws ExpansionException {

		HashMap stringKey2StringValue = new HashMap();
		for (Iterator entryIt = bibtexFile.getEntries().iterator();
			entryIt.hasNext();
			) {
			BibtexAbstractEntry abstractEntry =
				(BibtexAbstractEntry) entryIt.next();
			if (abstractEntry instanceof BibtexStringDefinition) {
				BibtexStringDefinition bibtexStringDefinition =
					(BibtexStringDefinition) abstractEntry;
				BibtexAbstractValue simplifiedValue =
					simplify(
						bibtexFile,
						bibtexStringDefinition.getValue(),
						stringKey2StringValue);
				bibtexStringDefinition.setValue(simplifiedValue);
				if (removeStringDefinitions) {
					bibtexFile.removeEntry(bibtexStringDefinition);
				};
				stringKey2StringValue.put(
					bibtexStringDefinition.getKey().toLowerCase(),
					simplifiedValue);

			}
			else if (abstractEntry instanceof BibtexPreamble) {
				BibtexPreamble preamble = (BibtexPreamble) abstractEntry;
				preamble.setContent(
					simplify(
						bibtexFile,
						preamble.getContent(),
						stringKey2StringValue));
			}
			else if (abstractEntry instanceof BibtexEntry) {
				BibtexEntry entry = (BibtexEntry) abstractEntry;
				for (Iterator fieldIt =
					entry.getFields().entrySet().iterator();
					fieldIt.hasNext();
					) {
					Map.Entry field = (Map.Entry) fieldIt.next();
					if (!(field.getValue() instanceof BibtexString)) {
						entry.addField(
							BibtexFile.makeField(
								(String) field.getKey(),
								simplify(
									bibtexFile,
									(BibtexAbstractValue) field.getValue(),
									stringKey2StringValue)));
					}
				}
			}
			else if (abstractEntry instanceof BibtexToplevelComment) {
				// don't do anything here ...	
			}
			else {
				throw new ExpansionException(
					"Expansions.expandStringReferences(): I don't support \""
						+ abstractEntry.getClass().getName()
						+ "\". Use the force, read the source!");
			}
		}
	}

	private static BibtexAbstractValue simplify(
		BibtexFile factory,
		BibtexAbstractValue compositeValue,
		HashMap stringKey2StringValue)
		throws ExpansionException {
		if (compositeValue instanceof BibtexString)
			return (BibtexString) compositeValue;
		if (compositeValue instanceof BibtexStringReference) {
			BibtexStringReference reference =
				(BibtexStringReference) compositeValue;
			String key = reference.getValue().toLowerCase();
			if (monthAbbreviations.contains(key))
				return reference;
			BibtexString simplifiedValue =
				(BibtexString) stringKey2StringValue.get(key);
			if (simplifiedValue == null)
				throw new ExpansionException(
					"Invalid string reference (target does not exist): \""
						+ reference.getValue()
						+ "\"");
			return simplifiedValue;
		}
		if (compositeValue instanceof BibtexConcatenatedValue) {
			BibtexConcatenatedValue concatenatedValue =
				(BibtexConcatenatedValue) compositeValue;
			BibtexAbstractValue left =
				simplify(
					factory,
					concatenatedValue.getLeft(),
					stringKey2StringValue);
			BibtexAbstractValue right =
				simplify(
					factory,
					concatenatedValue.getRight(),
					stringKey2StringValue);
			if (left instanceof BibtexString
				&& right instanceof BibtexString)
				return BibtexFile.makeString(
					((BibtexString) left).getValue()
						+ ((BibtexString) right).getValue());
			else
				return BibtexFile.makeConcatenatedValue(left, right);
		}
		throw new ExpansionException(
			"Expansions.simplify(): I don't support \""
				+ compositeValue.getClass().getName()
				+ "\". Use the force, read the source!");
	}

	/**
	 * Note: If you don't call expandStringReferences first, this function may lead to
	 * inconsistent string references.
	 * 
	 * @param file
	 */
	public static void expandCrossReferences(BibtexFile bibtexFile)
		throws ExpansionException {
		HashMap entryKey2Entry = new HashMap();
		ArrayList entriesWithCrossReference = new ArrayList();
		for (Iterator entryIt = bibtexFile.getEntries().iterator();
			entryIt.hasNext();
			) {
			BibtexAbstractEntry abstractEntry =
				(BibtexAbstractEntry) entryIt.next();
			if (!(abstractEntry instanceof BibtexEntry))
				continue;
			BibtexEntry entry = (BibtexEntry) abstractEntry;
			entryKey2Entry.put(entry.getKey().toLowerCase(), abstractEntry);
			if (entry.getFields().containsKey("crossref")) {
				entriesWithCrossReference.add(entry);
			}
		}
		for (Iterator entryIt = entriesWithCrossReference.iterator();
			entryIt.hasNext();
			) {
			BibtexEntry entry = (BibtexEntry) entryIt.next();
			String crossrefKey =
				((BibtexString) entry.getFields().get("crossref"))
					.getValue()
					.toLowerCase();
			entry.undefineField("crossref");
			BibtexEntry crossrefEntry =
				(BibtexEntry) entryKey2Entry.get(crossrefKey);
			if (crossrefEntry == null)
				throw new ExpansionException(
					"Crossref key not found: \"" + crossrefKey + "\"");
			if (crossrefEntry.getFields().containsKey("crossref"))
				throw new ExpansionException(
					"Nested crossref: \""
						+ crossrefKey
						+ "\" is crossreferenced but crossreferences itself \""
						+ ((BibtexString) crossrefEntry
							.getFields()
							.get("crossref"))
							.getValue()
						+ "\"");
			Map entryFields = entry.getFields();
			Map crossrefFields = crossrefEntry.getFields();
			for (Iterator fieldIt = crossrefFields.keySet().iterator();
				fieldIt.hasNext();
				) {
				String key = (String) fieldIt.next();
				if (!entryFields.containsKey(key)) {
					entry.addField(
						BibtexFile.makeField(
							key,
							(BibtexAbstractValue) crossrefFields.get(key)));
				}
			}
		}
	}

}
