/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "tcl_args.h"
 *									  created: 29/5/96 {12:24:55 am}	
 *								  last update: 05/22/98 {16:08:07 PM}	
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 * ===================================================================
 * Copyright (c) 1997  Vince Darley
 *  
 * See the file "license.terms" for information on usage and 
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * ===================================================================
 * ###################################################################
 */

/*@ ---------------------------------------------------------------------------
    ---------------------------------------------------------------------------
 */
#ifndef _Cpptcl_tcl_args_
#define _Cpptcl_tcl_args_

/*      \Section{Imports}
    Import required include files
 */
#include "cpptclInt.h"
#include "tcl_obj.h"
#include "tcl_strobj.h"

class tcl_base;

/* 
 * Note: this code works both with and without exception handling.
 * I currently avoid it, which means an	extra check	is required,
 * supplied	by the macro 'NO_EXCEPTIONS(arg)', which should
 * follow the 'arg >> done;' statement.	 The way the code is currently
 * set up, it could	be considered to trigger exception too early. 
 * Perhaps that	should be changed so '>> done' checks if an	error
 * occurred	and	triggers an	exception at that point	only.
 */

//@Section: Cpptcl library
//@Man: 
/** 
 * -------------------------------------------------------------------------
 *   
 * "class tcl_args_reader" --
 *  
 *  Simple class which holds a single element of an argument list, and
 *  allows you to read from it using nice '>>' operators, keeping a 
 *  record of what may have gone wrong.  Used internally; you only need
 *  to know about this if you write your own operators, and even then
 *  probably not.
 *  
 *  Reading, status, errors etc. are all handled in a manner similar to
 *  the iostreams.
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class tcl_args_reader {
	//template friend <class T> void operator>> (class tcl_args_reader& a, T& into);
  public:
	/// argument list item for Tcl8.x
	const Tcl_Obj* arg;
  public:
	///
  	enum _state { Good=0x0, Eof=0x1, Fail=0x2, Bad=0x4 } state;
	///
	tcl_args_reader(const Tcl_Obj* o=0):arg(o),state(Good) {}
	~tcl_args_reader(void) {}
	/// Is the state good?
	bool good(void) const {return (state ? false : true);}
	/// Is the state bad?
	bool bad(void) const {return ((state & 4) ? true : false);}
	/// Are we at eof?
	bool eof(void) const {return ((state & 1) ? true : false);}
	/// Did a conversion fail?
	bool fail(void) const {return ((state & 2) ? true : false);}

	#ifdef __GNUC__
	operator tcl_obj (void) const { return tcl_obj(arg);}
	#else
	///
	operator const tcl_obj& (void) const { return tcl_obj(arg);}
	#endif
	///
	operator tcl_obj (void) { return tcl_obj(arg);}
	///
	operator const Tcl_Obj* (void) const { return arg;}
	///
	operator Tcl_Obj* (void) { return (Tcl_Obj*)arg;}
	///
	void set_arg(const Tcl_Obj* the_arg) {arg = the_arg; state = Good;}
	///
	operator bool (void) {if(arg) return true; else {state = Eof;return false;}}
	///
	void setstate(tcl_args_reader::_state st) {state = st;}
	///
	void copy_arg(char* &dest) const;
	///
	void copy_arg_into(char* const &dest) const;
	///
	void examine_arg(const char* &dest) const;
	///
	void examine_arg(const Tcl_Obj* &dest) const;
};

//@ ----------------------------------------------------------------------------
/* \Section{Class Declaration}
 */

//@Section: Cpptcl library
//@Man: 
/** 
 *  "class tcl_args" --
 *  
 * A C++ class designed to deal effectively with C argument lists, of
 * the form 'int objc, Tcl_Obj ** argv', which are prevalent in C-Tcl
 * interaction.  
 * 
 * Backwards compatibility for old style C code which manipulates these
 * lists is maintained with minor code modifications
 * via the functions 'orig_argc(), orig_argv(int)', although it's best
 * if you just rewrite code to use the new features.
 * 
 * More C++'y usage is of the following form:
 * 
 *       tcl_object* rrandom::make_new(tcl_args& arg){
 *           arg >> arg.tmp_chars;
 *          
 *           if(!arg) {
 *               arg >> done; // so command completion works
 *               NO_EXCEPTIONS(arg,0); // required if no exceptions
 *               return new rrandom(arg,arg.tmp_chars);
 *           } else if(arg.syntax("value")=="seed") {
 *               int seed;
 *               arg >> seed >> done;
 *               NO_EXCEPTIONS(arg,0); // required if no exceptions
 *               rrandom *r = new rrandom(arg,arg.tmp_chars);
 *               r->Srand(seed);
 *               return r;
 *           } else {
 *               return 0;
 *           }
 *       }
 *      
 *       int rrandom::parse_tcl_command(tcl_args& arg){  
 *           if (!arg) { 
 *               arg >> done; // so command completion works
 *               NO_EXCEPTIONS(arg,TCL_ERROR);
 *               tcl_ << Rand() << result;
 *               return TCL_OK;
 *           } else if(arg.syntax("max")=="int") {
 *               int n;
 *               arg >> n >> done;
 *               NO_EXCEPTIONS(arg,TCL_ERROR);
 *               tcl_ << Randint(n) << result;
 *               return TCL_OK;
 *           } else
 *               return tcl_object::parse_tcl_command(arg);
 *       }
 *  
 * These two small functions will produce error messages of the following
 * informative form, with no further effort:
 *      
 *       % load libRandom
 *       % random
 *       random: No name or other arguments supplied.
 *       % random r
 *       r
 *       % random rr seed 123456
 *       read  partial cmd 'random rr seed' but failed to convert next 
 *       argument '123456' to type 'int'; syntax should be 
 *       'random rr seed value'
 *       % random rr seed 12345
 *       rr
 *       % rr
 *       0.525446
 *       % rr
 *       0.250916
 *       % rr
 *       0.893191
 *       % rr int 2345
 *       1349
 *       % rr int
 *       wrong # args: should be 'rr int max'
 *       % rr int 3456678 757
 *       read partial cmd 'rr int' but failed to convert next argument 
 *       '3456678' to type 'int'; syntax should be 'rr int max'
 *   
 * The really cool bit is that all the comparisons and syntax messages
 * which your C++ code contains are remembered by this class, so if an
 * error occurs in a Tcl script, a useful error message is generated
 * automatically.  Therefore the Tcl interface to your C++ code is self-
 * documenting!  
 *    
 * Command completion is also supported.  To support this facility,   
 * your class must make correct use of the 'arg >> done' command.  
 * Make sure (a) you always use 'done', and (b) you never perform
 * any action at all until after the 'done' command.
 * 
 * Finally unique abbreviations of sub-commands are also supported,
 * with some help from the programmer -- the code cannot itself
 * determine the unique abbreviations, but will match them if given.
 *    
 *   --Version--Author------------------Changes-------------------------------
 *     1.0     <darley@fas.harvard.edu> original
 *   -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class tcl_args {
  public:
	//@Man: Manipulators 
	//@{
	/** These are used to manipulate the state of the argument
	 *  parsing stream.
	 */
	/// Argument parsing complete. No more arguments should exist
	/** doc-done */
    friend tcl_args& done(tcl_args&);
	/// The next argument is optional
	/** doc-optional */
    friend tcl_args& optional(tcl_args&);
	/// Skip the next argument
	/** doc-skip */
    friend tcl_args& skip(tcl_args&);
	/** doc-skip */
    friend tcl_args& backup(tcl_args&);
	//@}
	
	/// This class's parse_tcl_command requires some inside information
	/// to work its magic.
	friend class cpptcl_info;
	
	/// Template friend is used for all read operations
	/** 
	 * This	would be very useful, but it doesn't work!?	However	it is in
	 * a recent	ANSI C++ draft,	so at some point I can use it and switch
	 * some	functions below	out	of this	class's	public interface.
	 */
	//template <class T> friend tcl_args& operator>>(tcl_args& arg, T& into);

  public:
	/// Main constructor.
	/** 
	 * The usual constructor --	it takes a tcl_obj for tcl-interaction,
	 * the argument	list it	obviously needs, and a tcl_object (whose name
	 * is entered into the 'parsed_so_far' list, if the object is given).
	 */
	tcl_args(tcl_obj&, int objc=0, Tcl_Obj* objv[]=0, const class tcl_base*o=0);

	/// Copy constructor.
	/** 
	 * Copy	constructor	-- perhaps if you're going to do something crazy with
	 * a tcl_args meta_object which doesn't belong to you, then make a copy and
	 * mess	with that.  This copy constructor only copies the arguments
	 * which haven't yet been used.
	 */
	tcl_args(const tcl_args&, const class tcl_base*o=0);
	virtual ~tcl_args(void);
	/// Manipulator
	//@See: done optional skip backup
	/** 
	 * To allow	tcl_args manipulation via 'done', 'optional', 'skip' and 'backup'
	 */
	tcl_args& operator >> (tcl_args& (*f)(tcl_args&));
	/// Casting operators
	//@{
	/** 
	 * These mean you can treat	me like	a tcl_obj/Tcl_Interp if you like
	 */
	///
	operator tcl_obj& (void) const;
	///
	operator Tcl_Interp* (void) const;
	///
	tcl_obj& tcl(void) const;
	///
	Tcl_Interp* interpreter(void) const;
	//@}

	//@Man: Argument syntax checking and string comparison
	//@{
	
	//@Man: Syntax and help texts
	//@{
	/** 
	 * Use any of these four to declare the correct
	 * syntax for the following	comparison with	'=='
	 */
	/// Declare the command syntax
	tcl_args& syntax(tcl_strobj syntax);
	/// Shorthand to declare command syntax
	tcl_args& operator()(tcl_strobj syntax);
	/// Declare command syntax and help text
	tcl_args& operator()(tcl_strobj syntax, tcl_strobj help_text, 
						 const char* abbrev=0);
	/// Declare desired type of next argument
	tcl_args& next_arg(Tcl_ObjType *t);
	tcl_args& next_arg(class cpx_type& t);
	tcl_args& next_arg(const char* const t);
	
	/// Declare command help text only
	/** 
	 * Attach some help	text for a following "-help" or "-h" query
	 */
	tcl_args& help(tcl_strobj help_text);
	//@}

	
	/// Is the next command item the given string?
	bool operator == (const char* const given_arg);
	/// Is the next command item the given string?
	bool operator == (const Tcl_Obj* given_arg);
	/// Always matches, and copies the matching pointer into the given_arg
	bool match_into(const char* &given_arg, const char* const arg_name);
	/// Always matches, and copies the matching pointer into the given_arg
	bool match_into(const Tcl_Obj* &given_arg, const char* const arg_name);
	/// Match next command item with a type
	bool is_a(Tcl_ObjType *t);
	/// Match next command item with a type
	bool is_a(class cpx_type& t);
	/// Match next command item with a type
	bool is_a(const char* const t);

	/** 
	 * Typical use of the above	six	functions is as	follows:
	 *		if(arg("my syntax","my help")=="cmd"||"equivalent cmd")...
	 */
	//@}

	/// Remove the given argument.  It must match.
	/** 
	 * Remove the given	argument.  Signals a syntax	error if
	 * the next	argument doesn't match exactly.
	 */
	tcl_args& operator -= (const char*);
	/// Remove the next argument
	//@See: skip
	/** 
	 * Remove the next argument	(same as 'skip'), signalling
	 * an error	if no such argument	exists.
	 */
	tcl_args& operator --(int);
	/// How many args left to parse?
	int args_left(void) const;
	/// What's the next argument?
	const char* peek(void) const;
	/// Is the next argument a real
	bool peek_is_float(void) const;
	/// Is the next argument an integer (fails if too big for int)
	bool peek_is_int(void) const;
	/// Is the next argument a long integer
	bool peek_is_long(void) const;
	/// Is the next argument a string (i.e. not a number)
	bool peek_is_string(void) const;
	/// Is the next argument the given string
	bool peek_is(const char* const is) const;
	
	//@Man: Remaining argument list manipulation
	//@{
	/** 
	 * These deal only with unprocessed arguments; they
	 * act like	!argc_left, argv[next], argc_left, 
	 * argv+next respectively.
	 */
	/// Are there any arguments left?
	bool empty(void) const;

	/// What's the n'th argument remaining?
	Tcl_Obj* operator[] (int n) const;

	/// How many arguments are left?
	operator int (void) const;

	/// Pointer to array of remaining arguments
	operator Tcl_Obj** (void) const;

	//@}
	//@Man: Original argument list manipulation
	//@{
	/** 
	 * These deal with the entire argument list; they
	 * act like	!argc, argv[x], argc, argv	respectively
	 */
	/// Are there any arguments at all?
	bool operator! (void) const;

	/// Returns '#objv[item]#'
	Tcl_Obj* orig_objv(int item) const;
	/// Returns '#objc#'
	int orig_objc(void) const;
	/// Returns '#objv#'
	Tcl_Obj** orig_objva(void) const;

	//@}

	/// No commands remain to test.  The user gave an illegal command.
	/** 
	 * There are no	more allowed matches to	what we've seen
	 * so far.	Triggers an	exception.
	 */
	int no_match(void);
	/// called internally by the toplevel Tcl interface.  don't use it.
	int no_match_yet(void);
	/// checks if you typed a unique abbrev.
	/** 
	 * usually only called by Cpptcl core code: 
	 */
	bool try_abbreviation(void);
	/// Deal with an objects `name'
	//@{
	/** 
	 * Since we	deal with Tcl objects and commands,	we create
	 * things with names.  Usually the user	supplies the name
	 * as the first	argument argv[0], but sometimes	we wish
	 * to over-ride	that choice, or	generate a unique name
	 * automatically.  Calling 'name' will return 'argv[0]',
	 * unless it was over-ridden with a	prior call to 'setName'.
	 */
	void setName(Tcl_Obj*);
	Tcl_Obj* check_name(void) const;
	/** 
	 * 'name' actually removes the name	from the argument
	 * list	by calling 'skip()'.  Use 'check_name()' if	you
	 * just	wish to	examine	the	name.
	 */
	Tcl_Obj* name(void);
	//@}
	
	///
	void signal_error(int err);
	/// Can be used to check if my internal stream is ok after a read
    bool arg_read_ok(void) const { return (my_args_.good() ? true: false); }
  public:
	//@Man: Items for external read functions
	/** 
	 * The following items are a part of my	interface which
	 * should only be used by 'operator	>>'	functions which
	 * extend my functionality.	 Once I	can	use	a template
	 * friend above, these will be made protected again.
	 */
	//@{
		// list of error types
		enum tcl_args_err { No_Err=0, Conversion=1, No_match=2, Syntax=3,
							Too_many_args=4, Cpp_constructor=5, Help=6,
						    Finding_completions=7 };
		// perform any post-read operations
		void read_done(void);
		// read a constant string from the argument list
		void const_string_read(const char*&);
		// inform me of the name of the thing that is trying to be read
		void set_conversion_type(Tcl_Obj* n) {
			Tcl_IncrRefCount(stored_type.obj = n); 
			type_of_stored_type = option_list;
		}
		// inform me of the name of the thing that is trying to be read
		void set_conversion_type(const char* n) {
			stored_type.str = n; type_of_stored_type = type_name;
		}
		
		//@Man: Error streams
		//@{
		/** 
		 * These are used internally to	allow clever generation
		 * of error	messages when parsing fails.
		 */
		/// what's been parsed (used for error messages)
		tcl_obj parsed_so_far;
		/// what went wrong (used for error messages)
		tcl_obj failed_args;
		//@}

		//@Man: Interface for templated read operations
		//@{
		 
        /* 
         * Similarly the following two can be used to create new read
         * functions such as the following:
         *     tcl_args& operator >> (tcl_args& arg, info_observable& into) {
         *         arg.set_conversion_type("info_observable");
         *         arg.check_read() >> into;
         *         arg.check_after();
         *         arg.parsed_so_far << into;
         *         arg.read_done();
         *         return arg;
         *     }
         */
		/// Read from stream, but check if there's anything to be read first
		tcl_args_reader& check_read(void);
		/// Check if the read went ok
		tcl_args& check_after(void);
		/// Have we been told this argument is optional?
		bool is_optional_arg(void) const;
		/// Just pretend to read (discard the argument)
		void pretend_read_from_me(void);
		//@}
	//@}
	
  protected:
	/// Storage for argument list
	tcl_args_reader my_args_;
	
	/// Name of current argument type -- used for error messages
	union {
		const char* str;
		Tcl_Obj* obj;
		Tcl_ObjType* tcl_type;
	} stored_type; 
	/// Is the above a type name or a list of itemised options
	enum { type_name = 0, option_list = 1, tcl_type = 2, cpptcl_type = 3
	} type_of_stored_type;
	/// Often the first argument is an object name which may be over-ridden
	Tcl_Obj* myName;
	/// Skip an argument --- the friend 'skip' is for external, public use
    void skip(void);
	/// Go back an argument --- the friend 'backup' is for external, public use
    void backup(void);
	/// The desired type of the next argument, zeroth bit is the optional flag
	mutable bool _optional_arg;
	/// Are we looking for a next arg, and return type
	bool desiring_next_arg_type(void) const;
	///
	bool next_arg_matches(void) const;
	/// Not sure?!
	void remove_from_buffer(void);
	
	/// We've got an error, with the given error code
	virtual void internal_signal_error(int err);
	
	//@Man: Internal methods to signal errors
	//@{
	/** 
	 * There are currently five	types of error,	signalled by the enum
	 * defined above, and each having a	different kind of error	message.
	 * They	should only	be called via their	wrapper	'signal_error'.
	 */
	/// Too many arguments were given, often called from 'done'
	void args_too_many_err(void);
	/// No command matched the given arguments
	void args_no_match_err(void);
	/// An argument failed to convert to the correct type
	void args_conversion_err(void);
	/// Too few arguments usually
	void args_syntax_err(void);
	/// ??
	void args_cpp_constructor_err(void);
	/// Not really an error. The user gave '-h' as an argument to ask for help
	/** 
	 * Finally there is a 'help-signal', if the user gives a
	 * '-h'	or '-help' argument, some help text	will be	returned if	the
	 * programmer kindly made it available.	 See 'help'	or 'operator()'
	 */
	void args_help_signal(void);
	//@}
	
	//@Man: Internal methods to deal with completions/help
	//@{
	/// 0=no, 1=yes, 2=long form, higher bits = help
    int _finding_completions;
	///
	int finding_completions(void) const { return _finding_completions & 3;}
	///
	Tcl_Obj* expansion;
	///
    enum completion_type_ { No_completion = 0, Unique_completion = 1, 
    	    Have_completion = 2, Already_complete = 3 };
    ///
    completion_type_ completion_type;
    ///
    int completion_orig_len;
    ///
    int completion_extra_len;
    ///
    tcl_obj completion;
	/// with_arguments = add command syntax to the completion
	void start_completion_check(bool with_arguments=false);
	///
	void end_completion_check(void);
	/// Used internally to handle command completion
	void args_completion_signal(void);
	///
	void remember_completion(const char* s);
	///
	bool finding_help(void) const { return _finding_completions & 4;}
	///
	bool help_flag(void) const;
	/// with_arguments = add command syntax to the completion
	void start_help_check(void);
	///
	void end_help_check(void);
	//@}

	/// Empty my arguments.  So I can be reused
    void clear_args(void);
    /// Put the given ostrstream back into pristine condition
	static void reset_stream(tcl_obj& o);
	
	/// My Tcl interpreter stream
	tcl_obj& tcl_;

	/// The number of arguments left to parse
	int args_left_;
	/// The command I'm currently comparing against
	Tcl_Obj* cur_cmd;
	/// The syntax of the current command
	tcl_strobj cmd_syntax;
	/// Help text for the current command
	tcl_strobj help_text;
	/// Already retrieved item from argument list
	bool got_cmd;
	/// smallest abbreviation for command
	const char* abbrev;
	
  private:
	int objc_;
	Tcl_Obj** objv_;
	int total_args(void) const {
		return objc_;
	}
	
	/// Used by 'operator==' and 'match_into'.
	bool _internal_match(const Tcl_Obj* &s, const char* const arg_name = 0);
	/// Used by 'operator==' and 'match_into'.
	bool _internal_match(const char* &s, const char* const arg_name = 0);
	/// Used by '_internal_match'
	bool _matching_equal(const Tcl_Obj* s, const Tcl_Obj* str);
	/// Used by '_internal_match'
	bool _failed_internal_match(const char* s);
	/// This always needs to be called before trying to match a command
	bool _ensure_have_command(void);
	/// Utility to turn any tcl meta_object into its string representation
	const char* object_to_string(const Tcl_Obj* obj) const;
	/// The caller must check that there are any args left first.
	const Tcl_Obj* peek_arg(void) const;

	/// The caller must check that there are any args left first.
	const char* peek_string_arg(void) const;
  public:
	/// This always needs to be called internally when we've matched a command
	/** 
	 * You should only call it yourself if you've peeked ahead in the argument
	 * list, and matched the next argument already. The parameter 'o' we add
	 * to the list we've parsed so far, so it should contain the textual string
	 * of the argument you just read.  If you don't give it, we just use that
	 * argument from our internal list.
	 */
	bool matched_cmd(const Tcl_Obj* o=0);
	/// 
	bool match_configuration_options;
	///
	static bool loose_match(const char* s1, const char* s2);
	/// store a container object for ease of creation
	mutable tcl_base* container;
	/// Are we doing command completion rather than real evaluation?
	bool completing(void) const { return (_finding_completions & 3 ? true : false);}
	
	/// Has an error already been signalled?
	int haveErr;
	int state(void) const {return haveErr;}
	/// the command we're currently checking against (with operator ==)
	const char* current_cmd(void) const;
	
	//@Man: Temporary read variables for external use
	//@{
	/** 
	 * These are useful	to avoid incessant declarations	of local variables,
	 * so rather than '#const char* n; arg >> n >> done; ...(n)...#', you can	
	 * just	use	'#arg >> arg.tmp_chars >> done;	...(arg.tmp_chars)...#'
	 */
	/// Constant string
	const char* tmp_chars;
	///
	int tmp_int;
	///
	long tmp_long;
	///
	float tmp_float;
	///
	double tmp_double;
	///
	bool tmp_bool;
	///
	char tmp_char;
	///
	Tcl_Obj* tmp_obj;
	//@}
	
};

//template <class T> tcl_args& operator>> (tcl_args& arg, T& into);

//@Section: Cpptcl library
//@Man: Reading from 'tcl_args'
//@{
/** 
 * This	is the basic 'read'	call which is used to extract arguments:
 * 
 *    template <class T> tcl_args& operator>> (tcl_args& arg, T& into);
 * 
 * I currently can't use the template _definition_ due to ambiguities.
 * However it is really	used behind	the	scenes for instantiation of these
 * functions.  The template	can	be used	to instantiate _any_ class which
 * reads correctly from an istream -- just '#\#include "tcl_args_t.cc"#'
 * when	it needs instantiating.
 */
///
tcl_args& operator >> (tcl_args& arg, char& v);
///
tcl_args& operator >> (tcl_args& arg, int& v);
///
tcl_args& operator >> (tcl_args& arg, long& v);
///
tcl_args& operator >> (tcl_args& arg, float& v);
///
tcl_args& operator >> (tcl_args& arg, double& v);

//@Man: Template specialisations for types we know more about:
//@{
///
tcl_args& operator >> (tcl_args& arg, const char*& i);
/// This just copies the pointer over: you must increment the refcount/duplicate
tcl_args& operator >> (tcl_args& arg, Tcl_Obj*& i);
///
tcl_args& operator >> (tcl_args& arg, char*& i);
///
tcl_args& operator >> (tcl_args& arg, bool& v);
///
tcl_args& operator >> (tcl_args& arg, class tcl_base*& into);
//@}

///
/**
 * Returns the name	of a given type.  When RTTI	is working fine,
 * we can probably replace this with some ANSI C++ rtti call.
 */
template <class T> const char* GETTYPE(T&);
//@}


/*
 * Include all inline functions	here.  We can do this directly in the header
 * file	because	this class doesn't interlock too heavily with others.
 */
#include "tcl_args.icc"

tcl_args_reader& operator>> (class tcl_args_reader& a, bool& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, int& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, long& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, float& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, double& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, char& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, const char*& into);
tcl_args_reader& operator>> (class tcl_args_reader& a, Tcl_Obj*& into);
//template <class T> tcl_args_reader& operator>> (class tcl_args_reader& a, T*& into);

#ifdef NO_EXCEPTION_HANDLING
#define CHECK_ARGS(a) \
	if(a.empty()) { \
		if(a.is_optional_arg()) a.pretend_read_from_me(); else a.check_read(); \
		return a; \
	}
#else
#define CHECK_ARGS(a) 
#endif

#endif
