/*
 * SOURCE:  memory.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: memory management, low level buffer operations
 *
 * UPDATES: 11/16/1991 - major rewrite
 *          03/11/1992 - bug in m_load() fixed
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

    /* The following manifest constants are used as flags in the
       undo buffer */
#define PM_DELETELINE (1<<0) /* the line has been deleted  */
#define PM_INSERTLINE (1<<1) /* the line has been inserted */
#define PM_BEGINGROUP (1<<2) /* the top of a group         */
#define PM_ENDGROUP   (1<<3) /* the end of a group         */
#define PM_BEGINLIST  (1<<4) /* the top of the list        */
#define PM_ENDLIST    (1<<5) /* the end of the list        */

    /* The following manifest constant declares the size of the
       undo buffer (e.g. the number of lines that can be undone */
#define RINGMEMSIZE 200

    /* The following structure is the basic element of the undo
       buffer */
static struct memblk {
    unsigned line;      /* The line number */
    unsigned long ptr;  /* The pointer to the freed line info */
    unsigned cx;        /* old CursorX */
    unsigned int flags; /* Flags for this element (s.o.) */
};

    /* The following array of structures is the undo buffer */
static struct memblk ring[RINGMEMSIZE];
    /* The following structure points to the current element
       in the undo buffer. This is the element the next entry
       is put to */
static struct memblk *currblk;
    /* The following variable is incremented each time either
       m_undo() or m_redo() is entered and decremented each time
       these functions exit. '_pushmem()' checks for positive
       values of 'restoring' and then exit immediately without
       doing anything. This prevents from pushing something to
       the undo buffer when undoing or redoing. */
static int restoring = 0;
    /* The following variable is set to PM_BEGINGROUP when
       m_beginundogroup() is called. When the next element of
       the undo buffer is filled (e.g. '_pushmem()' is called)
       the flags of this element are set to this value. */
unsigned int begingroup = 0;
    /* The following variable is used to store the CursorX
       position. */
unsigned int bgroupcx;

    /* Here is the (global) variable that holds basic information
       about the edit buffer. */
struct MemoryHeader mh;

    /* The following are declarations of variables that were defined
       somewhere else. */
extern struct Editor ed;
extern unsigned char currline[EDIT_WIDTH];
extern unsigned int videobase;
extern int verbose, status;
extern BOOL _redraw;
extern int contstat;

    /* local function prototypes */
static int backup_file( char * );
static void disk_full( char * );

static void _initringmem( void );
static void _pushmem( unsigned, unsigned long, unsigned int );
static void _removeundoinfo( unsigned long );
static struct memblk *_nextblk(struct memblk *);
static struct memblk *_prevblk(struct memblk *);

/*
 *    Name: m_init
 *  Return: memory size if successful, 0 if not.
 * Purpose: initializes memory and MemoryHeader.
 *
 *
 *
 * (c)M.Schollmeyer
 */
unsigned long m_init( void )
{
    unsigned int segment, maxmem;

    _initringmem();

    _asm {
        mov     ah, 48h        ; allocate imposible amount
        mov     bx, 0ffffh
        int     21h
        jc      nottoomuch
        mov     maxmem, 0ffffh ; strange, but true: 1024 kBytes free mem
        push    ax             ; free it again
        pop     es
        mov     ah, 49h
        int     21h
        jmp     donealloc
nottoomuch:
        mov     maxmem, bx     ; save the number of avaible paragraphs
donealloc:
    }
    /* maxmem now contains the maximum number of unfragmented paragraphs
       avaible from DOS. Maxmem should be at least MIN_MEM_SIZE
       plus 0x1000 bytes remaining to DOS */

    if( maxmem < (unsigned int)
        ( MIN_MEM_SIZE / BYTES_PER_PARAGRAPH ) + 0x1000 ) goto outmem;

    /* leave 0x1000 bytes to DOS */
    maxmem -= 0x1000;

    /* try to allocate this block */
    segment = AllocSeg( maxmem );
    if( segment == 0 ) goto outmem;

    /* setup MemoryHeader */

    /* size will be in bytes */
    mh.size = (unsigned long)maxmem * BYTES_PER_PARAGRAPH;

    FP_OFF( mh.base ) = 0;
    FP_SEG( mh.base ) = segment;

    mh.desc = (unsigned long _huge *)mh.base + (mh.size-1L)/sizeof(mh.desc);
    *(mh.desc) = 0L;

    mh.endlines = mh.base;

    return mh.size;

outmem:
    printf( errmsg( EM_NOEDITBUF|EM_MEMORY_C ) );

    return 0;
}



/*
 *    Name: m_cleanup()
 *  Return: TRUE (maybe success one day)
 * Purpose: cleans up memory management stuff
 *
 *
 *
 * (c)M.Schollmeyer
 */
int m_cleanup( void )
{
    if( mh.base )
        FreeSeg( FP_SEG( mh.base ) );

    return( 1 );
}




/*
 *    Name: m_getline
 *  Return: length of line read
 * Purpose: read any line from file buffer to specified buffer.
 *
 *
 *
 * (c)M.Schollmeyer
 */
int m_getline( unsigned int line, unsigned char _huge *buffer ) {

    unsigned char _huge *buf;     /* will point to line in memory soon */
    unsigned long _huge *desc;    /* will point to it's pointer        */
    int                 xpos = 0, /* needed to expand tabs             */
                        linelen;  /* will contain length of line       */

    desc = mh.desc - line;
    /* desc now points to the line pointer in my table */
    buf = mh.base + *desc + 2;
    /* buf now contains the pointer to the line in memory
    /* two bytes of line info are skipped */
    linelen = *(buf-1);

    /* Now we expand tabs */
    while( linelen ) {

        if( *buf == '\x09' ) {
            /* It's a tab char */
            ++buf;
            *buffer++ = ' ';
            ++xpos;
            /* play it save... */
            if( xpos + linelen + NUMTABS < EDIT_WIDTH ) {
                while( xpos % NUMTABS ) {
                    *buffer++ = ' ';
                    ++xpos;
                }
            }
        } else if( *buf == 0 ) {
            *buffer++ = ' ';
            ++buf;
            ++xpos;
        } else {
            *buffer++ = *buf++;
            ++xpos;
        }
        --linelen;
    }
    *buffer = '\0';
    return xpos;
}

/*
 *    Name: m_getcurrentline
 *  Return: void
 * Purpose: get current line from file buffer.
 *
 *
 *
 * (c)M.Schollmeyer
 */
void m_getcurrentline()
{
    m_getline( ed.CursorY, currline );
}



/*
 *    Name: m_putcurrentline
 *  Return: TRUE if successful, FALSE if not
 * Purpose: put current line to buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
BOOL m_putcurrentline( void ) {

    BOOL ret = TRUE;

    if( ed.Flags & LINECHANGED ) {
        ret = m_putline( currline, ed.CursorY, strlen( currline ) );
        CLEARBIT(ed.Flags, LINECHANGED);
    }

    return ret;
}



/*
 *    Name: m_putline
 *  Return: TRUE if successful, FALSE if not
 * Purpose: put any line to buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */

BOOL m_putline( unsigned char *cp, unsigned int line, register int len ) {

    unsigned char _huge *buf,
                        *end;
    unsigned long _huge *desc;

            /* first, get pointer to line */
    desc = mh.desc - line;

            /* Push the undo information to the undo buffer */
    _pushmem( line, *desc, 0 );

    len = _MIN( len, strlen( cp ) );

            /* if there is anything to put, check for trailing spaces */
    if( len ) {
        end = cp + len - 1;
        while( *end == '\0' || *end == ' ' ) {
            if( len == 0 )
                break;
            --end;
            --len;
        }
    }

            /* the line descriptor table grows from mh.desc,
             * the lines grow from mh.base. All we need to put
             * the current line is len + 2 bytes for flags and length.
             */
    if( (unsigned long)mh.desc - sizeof(unsigned char _huge *)
       * mh.numlines > (unsigned long)mh.endlines + len + 2 ) {
        /* there is enough space for new frame */
        /* free old pointer if it is already allocated */
        if( *desc != NULLPTR )
            *(*desc+mh.base) = FREEFRAME|SMALLFRAME;
        buf = mh.endlines;
        mh.endlines += len + 2;
    } else {
        /* not enough space */
        buf = mh.base;
        /* search for free frame */
        while( (*buf & ALLOCFRAME) || (*(buf+1) < (unsigned char)(len+2)) ) {
            buf += (unsigned long)*(buf+1) + 2L;
            if( buf >= mh.endlines ) {
                DoErrorBox( HLP_PUTLINE, errmsg(EM_OUTOFMEM) );
                /* should pack here... */
                _redraw = TRUE;
                return FALSE;
            }
        }
        /* now we found a frame with a length of more than len+2:
         * 2 bytes for line info.
         */
        /* free old pointer if it is already allocated */
        if( *desc != NULLPTR ) {
            _removeundoinfo( *desc );
            *(*desc+mh.base) = FREEFRAME|SMALLFRAME;
        }
        *(buf+1) -= len+2;
        *buf = FREEFRAME|SMALLFRAME;    /* mark as free */
        buf += (unsigned long)*(buf+1) + 2L;
    }

    *desc = buf - mh.base;
    *buf++ = ALLOCFRAME|SMALLFRAME;
    *buf++ = (unsigned char)len;

    while( len ) {
        --len;
        *buf++ = *cp++;
    }

    SETBIT(ed.Flags, MODIFIED);

    return TRUE;
}


/*
 *    Name: m_insertline
 *  Return: TRUE if successful, FALSE if not
 * Purpose: insert one line at specified line number
 *
 *
 *
 * (c)M.Schollmeyer
 */
BOOL m_insertline( unsigned int line ) {

    unsigned long _huge *desc;
    register unsigned int cl = mh.numlines - line + 1;

            /* get new descriptor pointer */
    desc = mh.desc;
    desc -= mh.numlines+1;
            /* desc now points to the first free entry */

            /* if we are above marker, shift marker y-koord */
    if( line <= ed.MarkY ) ++ed.MarkY;

            /* since desc is casted to (char _huge *) there must
             * be at least 4 bytes for a _huge pointer
             */
    if( (unsigned char _huge *)desc > mh.endlines+3 ) {
        /* there's enough space for new pointer.
         * Now copy all pointer up one line
         */
        while( cl ) {
            *desc = *(desc+1);
            ++desc;
            --cl;
        }

        /* mark pointer as unassigned */
        *desc = NULLPTR;
        /* The MODIFIED flag is set because we changed the buffer. The
           LINECHANGED flag is set so e_up or e_down will store the line,
           there can always be only one line assigned to 0L
         */
        SETBIT( ed.Flags, MODIFIED|LINECHANGED );
        ++mh.numlines;

        _pushmem( line, *desc, PM_INSERTLINE );
        return TRUE;
    }

    _redraw = TRUE;
    DoErrorBox( HLP_PUTLINE, errmsg(EM_OUTOFMEM) );
    return FALSE;
}


/*
 *    Name: m_deleteline
 *  Return: void
 * Purpose: delete any number of lines form buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
void m_deleteline( unsigned int line, unsigned int num ) {

    unsigned long   _huge *desc,
                    _huge *scan;
    register unsigned int  begin, i;

    begin = line + num;
    if( begin > mh.numlines ) {
        begin = mh.numlines;
        num = mh.numlines - line + 1;
    }

    /* get line descriptor pointer */
    desc = mh.desc;
    desc -= line;

    if( line < ed.MarkY ) ed.MarkY -= num;

    for( i = num, scan = desc; i; --i ) {
        if( *scan != NULLPTR ) {
            _pushmem( line, *scan, PM_DELETELINE );
            /* mark line as free */
            *(*scan+mh.base) = FREEFRAME|SMALLFRAME;
        }
        --scan;
    }

    for( i = mh.numlines - begin + 1; i; --i, --desc )
        *desc = *(desc-num);

        /* this entry is not used usually, could be necessary if
           ed.CursorY == mh.numlines or so...
         */
    *desc = NULLPTR;

    SETBIT( ed.Flags, MODIFIED );
    mh.numlines -= num;
}


/*
 *    Name: m_load
 *  Return: TRUE if successful, FALSE if not, CANCEL if all action
 *          should be aborted. The calling function should take care of the
 *          edit buffer then since it could be undefined then.
 * Purpose: loads a file to buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
/*
    m_load {
        open file;
        scan file {
            if(next block doesn't fit) {
                query user;
                abort if necessary;
                copy last line to top of buffer;
                reinitialize buffer;
            }
            read next block;
            scan block {
                scan line;
                if(next line doesn't fit) {
                    query user;
                    abort if necessary;
                    copy rest of block to top of buffer;
                    reinitialize buffer;
                    continue;
                }
                put line to buffer;
            }
        }
    }
*/
BOOL m_load( fptr )
int fptr;
{
    unsigned int                 nlines;
    int                          linelen,
                                 i;
    unsigned char                s[EDITW_WIDTH],    /* general purpose buffer */
                          _huge *buf,       /* points to the current position */
                          _huge *line,      /* points to top of line */
                          _huge *endlines;  /* points to end of lines */
    unsigned long         _huge *desc;      /* points to current position in
                                               line descriptor table */
    long                         filesize,  /* total size of file, read by
                                               Seek() */
                                 scan,      /* goes from filesize to 0L and
                                               determines how much of the file
                                               has already been read */
                                 len;       /* goes from 0xffffL to 0L and
                                               determines how much of the buffer
                                               been read has been converted into
                                               lines */
    static char *file_too_large = "File too large, truncated.\nLoad next block?";

    /* initialize the undo buffer */
    _initringmem();

    /* get total size of file (and rewind it!) */
    filesize = Seek( fptr, 0L, SEEK_END );
    Seek( fptr, 0L, SEEK_START );

    /* initialize scan */
    scan = filesize;

    /* from now on we load a part of the file and convert it into lines */

    /* initialize buffer */
    desc = mh.desc;
    endlines = mh.base;
    line = mh.base;
    buf = mh.base + 2;
    linelen = 0;
    nlines = 0;

    do {   /* scan file */

        /* pass 1: read file to buffer */
        if( scan < 0xffff )
            len = scan;
        else
            len = 0xffff;

        /* look if this buffer fits into memory */
        while( (buf+len) >= (unsigned char _huge *)desc ) {
            i = DoMessageBox( DMB_YES|DMB_NO|DMB_CANCEL, HLP_FILETOOLARGE,
                 lpszWarning, file_too_large );
            if( i == DMB_NO )
                goto done;
            if( i == DMB_CANCEL ) {
                /* abort load process */
                Close( fptr );
                e_status( 1 );
                return CANCEL;
            }
            /* now the user wants to read the next block into memory.
               copy the last line wich has not been setup yet to top of buffer */
            buf = mh.base;
            /* line points to the last line */
            while( line < endlines )
                *buf++ = *line++;

            line = mh.base;
            desc = mh.desc;
            endlines = buf;
            nlines = 0;
        }

        /* read next block (if any) */
        if( len ) {
            len = (long)_Read( fptr, buf, (unsigned)len );
            if( len == 0 ) {
                DoErrorBox( HLP_DEFAULT, "Error reading file." );
                Close( fptr );
                e_status( 1 );
                return FALSE;
            }
        }
        /* pass 2: setup linedesc and make frames */

        /* adjust scan by the bytes beeing read */
        scan -= len;
        endlines = buf + len;

        /* this makes our line scan algorithm succeed in any case */
        endlines[0] = 0x0d;

        /* if linelen!=0 there is a previous uncomplete line wich should be
           rescanned, blocks might have been split up at "\x0d\x0a" */
        buf = line + 2;
        len += linelen;

        /* scan the buffer */
        do {
            /* get next line */
            while( *buf != 0x0d ) ++buf;
            /* check if line fits into buffer (the linedesc table could have
               grown too much) */
            if( buf+sizeof(unsigned char _huge *)
                >= (unsigned char _huge *)desc ) {
                /* This line does not fit into memory */
                i = DoMessageBox( DMB_YES|DMB_NO|DMB_CANCEL, HLP_FILETOOLARGE,
                     lpszWarning, file_too_large );
                if( i == DMB_NO )
                    goto done;
                if( i == DMB_CANCEL ) {
                    /* abort load process */
                    Close( fptr );
                    e_status( 1 );
                    return CANCEL;
                }
                scan += endlines - line;
                /* copy all remaining lines to top of buffer */
                for( buf = mh.base; line < endlines; ++line )
                    *buf++ = *line++;
                /* reinitialize buffer */
                endlines = buf;
                nlines = 0;
                continue;
            }

            if( buf-(line+2) > EDIT_WIDTH-1 ) {
                /* file has long line */
                buf = line+2+(EDIT_WIDTH-1);
            }
            linelen = buf-(line+2);
            len -= linelen;

            if( nlines % 50 == 0 ) {
                sprintf( s, " (%u) ", nlines );
                VideoStr( CR_STATUS, s, VideoBase->ScreenWidth-8, VideoBase->ScreenHeight-1 );
            }

            if( (len <= 0L) && (scan > 0L) ) {
                /* There is nothing more to do. We reached the end of this
                   read session. We don't add this to the buffer yet */
                break;
            }

            /* add line to buffer */
            *line = ALLOCFRAME|SMALLFRAME;
            *(line+1) = (unsigned char)linelen;
            *desc-- = (unsigned long)(line - mh.base);
            linelen = 0;
            line = buf;

            ++nlines;

            /* skip '\x0d\x0a' */
            buf += 2;
            len -= 2;
        } while( len > 0L );
    } while( scan > 0L );
done:
    mh.numlines = nlines-1;
    mh.endlines = endlines;

    CLEARBIT( ed.Flags, MODIFIED|LINECHANGED|PSEUDOFILE|TEMPFILE|MARKED );
    e_drawflags();

    return TRUE;
}


/*
 *    Name: m_loadvirtual
 *  Return: TRUE if successful, FALSE if not, CANCEL if all action
 *          should be aborted
 * Purpose: reloads a file of the file list from the corresponding
 *          temporary file.
 *
 *
 * (c)M.Schollmeyer
 */
BOOL m_loadvirtual( unsigned slot ) {

    char                *file,
                        *rfile,
                         buffer[SCREEN_WIDTH],
                        *cp;
    int                  fptr,
                         fptr2,
                         i,
                         len;
    unsigned char _huge *buf;
    unsigned long        vdate,
                         rdate,
                         size;
    unsigned char        recover = 0;
                                   /* flag set to 1 if the file should be
                                   recovered from the virtual file */
    struct MemoryHeader  mhead;

    /* get file names */
    file = GetProfileString( PR_VIRTFILE( slot ), NULL );
    rfile = GetProfileString( PR_FILENAME( slot ), NULL );

    /* there is no temporary file, this can happen if we reload a file
       at the very beginning of our session, or the temporary file didn't
       fit on disk or so */
    if( !file )
        return FALSE;

    buf = mh.base;

    /* try to open temporary file */
    fptr = Open( file, ACCESS_READ );
    if( fptr == -1 ) {
        /* open failed */
        PutError( -1, file );

        /* we couldn't open the temporary file, clear its file name.
           This is necessary because the following call to e_visit()
           from m_reload() will call m_savevirtual() wich first tries
           to reopen the temporary file */

        cp = GetProfileString( PR_VIRTFILE(slot), NULL );
        if( cp ) {
            ClearProfileData( PR_VIRTFILE(slot) );
            Delete( cp );
        }

        return FALSE;
    }

    fptr2 = Open( rfile, ACCESS_READ );
    if( fptr2 == -1 ) {
        i = GetExtError();
        if( i == 0x53 || i == 0x03 || i == 0x15 ) {
            PutError( i, rfile );
            Close( fptr );
            return CANCEL;
        }
        MakeShortName( buffer, rfile, 40 );
        i = DoMessageBox( DMB_YES|DMB_NO, HLP_RECOVER,
                 lpszWarning,
                "'%s'\nFile has been deleted.\nRecover?", buffer );
        if( i == DMB_NO ) {
            Close( fptr );
            Delete( file );
            file[0] = '\0';
            return CANCEL;
        }
        recover = 1;

    } else {

        vdate = GetFileDate( fptr );    // get date of virtual file
        rdate = GetFileDate( fptr2 );   // get date of real file
        Close( fptr2 );
        if( vdate != rdate && vdate && rdate ) {
            MakeShortName( buffer, rfile, 40 );
            i = DoMessageBox( DMB_YES|DMB_NO, HLP_REFRESH,
                              "Message",
                              "'%s'\nFile has been modified.\nRefresh?",
                              buffer );
            if( i == DMB_YES ) {
                Close( fptr );
                /* This will reload the file from disk */
                return FALSE;
            }
            if( i == DMB_CANCEL ) {
                Close( fptr );
                return CANCEL;
            }
        }
    }

    _Read( fptr, (char _far *)&mhead, sizeof( mhead ) );

    /* calculate size of virtual file */
    size = mhead.endlines - mhead.base;
    if( size + 4*mhead.numlines > mh.size ) {
        /* unable to reload, try to load real file again */
        Close( fptr );
        return FALSE;
    }

    /* setup MemoryHeader */
    mh.endlines = mh.base + size;
    mh.numlines = mhead.numlines;

    len = _Read( fptr, (char _far *)ring, sizeof( ring ) );
    if( len != sizeof( ring ) ) goto error;
    len = _Read( fptr, (char _far *)&i, sizeof( unsigned ) );
    if( len != sizeof( unsigned ) ) goto error;
    currblk = ring + i;

    while( (buf + 0x8000L) < mh.endlines ) {
        len = _Read( fptr, buf, 0x8000 );
        if( len != 0x8000 ) goto error;
        FP_SEG(buf) += 0x800;
    }

    _Read( fptr, buf, (unsigned)(mh.endlines - buf) );
    _Read( fptr, (char _far *)(mh.desc - mh.numlines), (mh.numlines+1) * 4 );

    Close( fptr );

    CLEARBIT( ed.Flags, (MODIFIED|LINECHANGED|PSEUDOFILE|TEMPFILE|MARKED) );
    if( !recover && (GetFileAttr( rfile ) & ATTR_READONY) )
        SETBIT( ed.Flags, READONLY );
    else
        CLEARBIT( ed.Flags, READONLY );

    e_drawflags();

    MakeShortName( ed.ShortFileName, rfile, SHORTFILENAMELEN );
    if( ed.DiskFileName ) free( ed.DiskFileName );
    ed.DiskFileName = HeapStrDup( rfile );
    if( slot != ed.CurrFile ) {
        ed.LastFile = ed.CurrFile;
        ed.CurrFile = slot;
    }

    if( recover )
        m_save( SAVE_ALWAYS );

    return TRUE;

error:
    Close( fptr );
    return FALSE;
}


static unsigned char _far *_buffer, _far *_bufptr;
static unsigned _size;

static BOOL _writeln( int, unsigned char *, unsigned );
static BOOL _flushbuf( int );

// #pragma optimize( "egl", off )
int m_save( int flag )
{
    int                fptr = -1,
                       i,
                       ret = 0;
    char              *buffer = NULL,
                      *file = NULL,
                       sfilename[SCREEN_WIDTH],
                       spaces[NUMTABS+1];
    unsigned int       line,
                       xpos,
                       len,
                       err;
    unsigned char     *cp,
                      *dst;

    for( i = 0; i < NUMTABS; ++i )
        spaces[i] = ' ';

    /* if the file has not been modified, and we want to save
       normal only, return 'OK' */
    if( ! ( ed.Flags & (MODIFIED|LINECHANGED) ) ) {

        if( flag == SAVE_NORMAL )
            return 1;

    } else if( ! (ed.Flags & AUTOSAVE) && flag == SAVE_NORMAL ) {

        i = _saverequest( ed.ShortFileName );
        if( i == CANCEL )       // responsed 'cancel'
            return 0;
        if( i == FALSE )        // responsed 'no'
            return 1;
    }

    /* allocate buffers */
    buffer = AllocHeap( 2 * EDIT_WIDTH );
    if( buffer == NULL )
        return 0;

    file = AllocHeap( BUFSIZE );
    if( file == NULL )
        goto error;

    if( ed.DiskFileName )
        strcpy( file, ed.DiskFileName );
    else
        file[0] = '\0';

    i = 0;

    if( (ed.DiskFileName == NULL ) || (flag == SAVE_QUERY) ) {
        /* no file name for this buffer or flag set, request */
        while( 1 ) {

            DOS_Request( "Save File", file, ed.DiskFileName, HLP_SAVEFILE );

            if( file[0] == '\0' ) {
                /* means DOS_Request() cancelled */
                goto error;
            }

            /* look, if it's the same name we already have... */
            if( ed.DiskFileName && strcmp( file, ed.DiskFileName ) != 0 ) {
                /* It's not the same, look if file exists */
                if( _Access( file, 0 ) == TRUE ) {

                    MakeShortName( sfilename, file, VideoBase->ScreenWidth-7 );
                    len = DoMessageBox( DMB_YES|DMB_NO|DMB_CANCEL, HLP_OVERWRITE,
                                      lpszWarning, "'%s'\nalready exists.\nOverwrite?",
                                      sfilename );

                    if( len == DMB_YES ) {
                        i = backup_file( file );
                        break;
                    }
                    if( len == DMB_NO )
                        continue;
                    /* len == DMB_CANCEL */
                    goto error;
                }
            } else
                i = backup_file( ed.DiskFileName );
            break;
        }
    } else
        i = backup_file( ed.DiskFileName );

    if( i ) {
        PutError( i, ed.DiskFileName );
        goto error;
    }

    for( _size = 0xffff, _buffer = NULL ;; _size -= 0x1000 ) {
        if( _buffer = malloc( _size ) )
            break;
        if( _buffer == NULL && _size <= 0x1000 ) {
            DoErrorBox( HLP_DEFAULT, errmsg(EM_OUTOFMEM) );
            goto error;
        }
    }
    _bufptr = _buffer;

retry:
    fptr = Create( file, 0 );
    if( fptr == -1 )
    {
        if( PutError( -1, file ) == DMB_RETRY )
            goto retry;
        goto error;
    }

    if( verbose > 0 ) {
        MakeShortName( sfilename, file, VideoBase->ScreenWidth-21 );
        m_putmsg( MSG_STNDRD, "Saving %s...", sfilename );
        status = 1;
    }

    m_beginundogroup();
    i = m_putcurrentline();
    m_endundogroup();
    if( !i ) goto error;

    for( line = 0; line <= mh.numlines; ++line ) {
        if( line % 10 == 0 ) {
            m_putxmsg( CR_STATUS, VideoBase->ScreenWidth-8, " (%u) ", line );
        }

        cp = buffer + EDIT_WIDTH;
        dst = buffer;

        if( ed.Flags & EXPANDTABS ) {
            len = m_getline( line, buffer );
            buffer[len++] = '\x0d';
            buffer[len++] = '\x0a';
        } else {
            len = m_getline( line, cp );
            for( xpos = 0; xpos < len; ) {
                if( *cp == ' ' ) {
                    i = NUMTABS - xpos % NUMTABS;
                    if( !strncmp( cp, spaces, i ) && i > 1 ) {
                        *dst++ = '\x09';
                        cp += i;
                        xpos += i;
                    } else {
                        *dst++ = ' ';
                        ++cp;
                        ++xpos;
                    }
                } else {
                    *dst++ = *cp++;
                    ++xpos;
                }
            }

            *dst ++ = '\x0d';
            *dst    = '\x0a';
            len = dst - buffer + 1;
        }
        if( _writeln( fptr, buffer, len ) != TRUE ) {
            err = GetExtError();
            if( err == 0 ) {
                disk_full( file );
            } else {
                ret = PutError( -1, file );
                if( ret == DMB_RETRY )
                    goto retry;
            }
            Close( fptr );
            fptr = -1;
            Delete( file );
            goto error;
        }
    }

    for ever {
        if( _flushbuf( fptr ) == FALSE ) {
            err = GetExtError();
            if( err == 0 ) {
                disk_full( file );
            } else {
                ret = PutError( -1, file );
                if( ret == DMB_RETRY )
                    continue;
            }
            Close( fptr );
            fptr = -1;
            Delete( file );
            goto error;
        } else
            break;
    }

    Close( fptr );
    fptr = -1;

    CLEARBIT(ed.Flags, MODIFIED|LINECHANGED|MARKED );

    ret = 1;

    if( ! (ed.Flags & PSEUDOFILE) ) {
        PutProfileString( PR_FILENAME( ed.CurrFile ), file );
    }
    if( ed.DiskFileName ) free( ed.DiskFileName );
    ed.DiskFileName = HeapStrDup( file );
    MakeShortName( ed.ShortFileName, file, SHORTFILENAMELEN );

    m_savevirtual();

error:

    if( fptr != -1 ) Close( fptr );

    if( buffer ) free( buffer );
    if( _buffer ) free( _buffer );
    if( file )   free( file );

    e_status( 1 );

    return ret;
}
// #pragma optimize( "", on )

static BOOL _writeln( int fptr, unsigned char *cp, unsigned len ) {

    BOOL ret = TRUE;

    if( _buffer + _size < _bufptr + len ) {

        /* flush buffer */
        ret = _flushbuf( fptr );

    }

    if( ret ) {
        memcpy( _bufptr, cp, len );
        _bufptr += len;
    }

    return ret;
}

static BOOL _flushbuf( int fptr ) {

    unsigned size;

    size = (unsigned)( _bufptr - _buffer );
    if( size ) {
        if( _Write( fptr, _buffer, size ) != size )
            return FALSE;
        _bufptr = _buffer;
    }
    return TRUE;
}

extern int errnum; /* defined in dos.c */

static int backup_file( char *name ) {

    char new[_MAX_DIR+_MAX_FNAME+_MAX_EXT],
         path[_MAX_DIR+1],
         base[_MAX_FNAME+1],
         ext[_MAX_EXT+1];

    if( ed.Flags & BACKUP ) {
        if( _Access( name, ACCESS_WRITE ) == TRUE ) {
            /* file exists */
            splitpath( name, path, base, ext );
            if( stricmp( ext, "bak" ) == 0 )
                return 0;
            sprintf( new, "%s%s.bak", path, base );
            Delete( new );
            return Rename( name, new );
        }

        errnum = 0x02;
        return 0; /* file not found */
    }
    return 0;
}


static void disk_full( char *file ) {

    char msg[SCREEN_WIDTH];

    MakeShortName( msg, file, VideoBase->ScreenWidth-7 );
    DoErrorBox( HLP_DISKFULL, "Write Error on file\n'%s'\nDisk Full?", msg );
}


int m_savevirtual( void ) {

    char tmpfilename[BUFSIZE];
    unsigned char _huge *buf;
    int fptr;
    char *file, *rfile;
    unsigned long date;
    unsigned int len, i;

    if( ed.Flags & PSEUDOFILE )
        return 0;

    file = GetProfileString( PR_VIRTFILE( ed.CurrFile ), NULL );
    rfile = GetProfileString( PR_FILENAME( ed.CurrFile ), NULL );

    if( rfile ) {
        fptr = Open( rfile, ACCESS_READ );
        if( fptr == -1 ) {
            PutError(-1, rfile );
            return 0;
        }
        date = GetFileDate( fptr );
        Close( fptr );
    } else {
        /* This should never happen: we have a virtual file name
           but no corresponding real one */
        DoErrorBox( HLP_SAVVIRT1, "Internal error\n'%s@%d'", __FILE__, __LINE__ );
        return 0;
    }

    buf = mh.base;

    if( file ) {
        strcpy( tmpfilename, file );
        fptr = Open( tmpfilename, ACCESS_WRITE );
        if( fptr == -1 ) {
            PutError(-1, tmpfilename );
            ClearProfileData( PR_VIRTFILE( ed.CurrFile ) );
            return 0;
        }
    } else {
        fptr = CreateTmp( tmpfilename, 0 );
        if( fptr == -1 ) {
            DoErrorBox( HLP_SAVVIRT2, "Could not create virtual file\n'%s'",
                        tmpfilename );
            return 0;
        }
    }


    len = _Write( fptr, (char _far *)&mh, sizeof( mh ) );
    if( len != sizeof( mh ) ) goto error;

    len = _Write( fptr, (char _far *)ring, sizeof( ring ) );
    if( len != sizeof( ring ) ) goto error;
    i = (unsigned)(currblk - ring);
    len = _Write( fptr, (char _far *)&i, sizeof( unsigned ) );
    if( len != sizeof( unsigned ) ) goto error;

    while( (buf + 0x8000L) < mh.endlines ) {
        len = _Write( fptr, buf, 0x8000 );
        if( len != 0x8000 ) goto error;
        FP_SEG(buf) += 0x800;
    }

    len = (unsigned)(mh.endlines - buf);
    if( _Write( fptr, buf, len ) != len )
        goto error;

    len = (mh.numlines+1) * 4;
    if( _Write( fptr, (char _far *)(mh.desc - mh.numlines), len ) != len )
        goto error;

    PutProfileString( PR_VIRTFILE( ed.CurrFile ), tmpfilename );

    SetFileDate( date, fptr );
    Close( fptr );

    return 1;

error:
    Close( fptr );
    Delete( tmpfilename );
    return 0;
}


/*
 *    Name: m_pack
 *  Return: always 0 (makes edit() keep the message)
 * Purpose: packs the edit buffer
 *
 *
 *
 * (c)M.Schollmeyer
 */
#include <string.h>

int m_pack( int key, int c ) {

    unsigned char _huge *src, _huge *dst, _huge *buf, _huge *end;
    unsigned long _huge *desc;
    unsigned long len, disp = 0L, scan;
    register unsigned int i;

    dst = mh.base;

    if( !m_putcurrentline() ) {
        m_putmsg( MSG_ERROR, "pack not avaible." );
        return 0;
    }

    /* search for first free frame */
    for( end = mh.endlines, src = dst; src < end; dst+= *(dst+1) + 2 ) {
        if( *dst == (SMALLFRAME|FREEFRAME) )
            break;
    }

    /* now dst is assigned to the found free frame */
    for( src = dst; src < end; ) {
        disp = 0L;
        len = 0L;
        // search for the next allocated frame
        while( *src != (SMALLFRAME|ALLOCFRAME) && src < end ) {
            /* this one is free (or lost), skip it */
            src += *(src+1) + 2L;
        }
        /* now src points to the next allocated frame
           search for next free frame, redirect pointers */
        disp = src - dst;
        for( buf = src; buf < end; ) {
            if( *buf == (SMALLFRAME|ALLOCFRAME) ) {
                // this one is allocated, search for pointer
                desc = mh.desc;
                scan = buf - mh.base;

                for( i = mh.numlines + 1; i; --i, --desc ) {
                    if( *desc == scan ) {
                        // redirect this one
                        *desc -= disp;
                        break;
                    }
                }
                len += *++buf + 2;
                buf += *buf   + 1;
            } else
                break;
        }

        // Now we can copy from <src> to
        // <dst>, <len> bytes.
        /* copy mem */
        while( len ) {
            *dst++ = *src++;
            --len;
        }
    }

    // adjust end of buffer
    mh.endlines = dst;
    _initringmem();
    m_putmsg( MSG_STNDRD, "pack done." );
    return 0;
}


unsigned int m_framesize( unsigned int line ) {

    unsigned long _huge *desc;

    desc = mh.desc;
    desc -= line;

    return (unsigned int)(*(mh.base+*desc+1) );
}

unsigned long MemFree( void ) {

    unsigned free = 0;
    unsigned long freebytes;
    BOOL scan = TRUE;
    unsigned numblks = 0;

    _fheapmin();

    while( scan ) {
        _asm {
            mov ah, 48h
            mov bx, 0ffffh
            int 21h
            jnc toomuch
            cmp bx, 0
            jne scanon
            mov scan, 0
            jmp endloop
scanon:
            mov ah, 48h
            int 21h
toomuch:
            push ax
            inc  numblks
            add free, bx
endloop:
            nop
        }
    }

    while( numblks ) {
        _asm {
            pop es
            mov ah, 49h
            int 21h
        }
        --numblks;
    }

    freebytes = (unsigned long)free * BYTES_PER_PARAGRAPH;

    freebytes += _memavl();

    return freebytes;
}

/*
 *    Name: AllocHeap, HeapStrDup
 *  Return: a pointer to the requested memory or NULL if unsuccessful.
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
void *AllocHeap( unsigned int size ) {

    void *ptr = NULL;

    if( size < 64 ) {
        ptr = _nmalloc( size );
    }

    if( ptr == NULL ) {
        /* either tiny allocation failed or huge item */
        ptr = _fmalloc( size );
    }

    if( ptr == NULL ) {
        /* failed, display error */
        DoErrorBox( 0, errmsg(EM_NOCORE) );
    }

    return ptr;
}

void *HeapStrDup( char *str ) {

    char *ptr;

    if( ! ( ptr = strdup( str ) ) ) {
        DoErrorBox( 0, errmsg(EM_NOCORE) );
    }

    return ptr;
}


/*
 *
 * Stack Functions
 *
 *
 */

struct StackPos {
    struct StackPos *prev;
    UBYTE data[];
};

void FreeStack( void *sp ) {

    while( sp = PopStack( sp ) );
}

void *PopStack( void *sp ) {

    struct StackPos *stp = (struct StackPos *)((APTR)sp-sizeof(struct StackPos));

    if( sp ) {
        if( stp->prev ) sp = (void *)((APTR)(stp->prev)+sizeof(struct StackPos));
        else            sp = NULL;
        free( stp );
    }

    return sp;
}

void *PushStack( void *sp, UBYTE *data, unsigned size ) {

    struct StackPos *new = AllocHeap(sizeof(struct StackPos)+size);

    if( new ) {
        if( sp )
            new->prev = (struct StackPos *)((APTR)sp-sizeof(struct StackPos));
        else
            new->prev = NULL;

        memcpy( new->data, data, size );
        return( new->data );
    }
    return NULL;
}


static void _initringmem( void ) {

    currblk = ring;
    SETBIT( currblk->flags, PM_BEGINLIST|PM_ENDLIST );
    begingroup = 0;
}

static void _pushmem( unsigned line, unsigned long ptr, unsigned int flags ) {

    if( restoring ) return;

    /* advance currblk */
    CLEARBIT( currblk->flags, PM_ENDLIST );
    currblk = _nextblk( currblk );

    if( currblk->flags & PM_BEGINLIST )
        SETBIT( _nextblk( currblk )->flags, PM_BEGINLIST );

    currblk->line = line;
    currblk->ptr = ptr;
    currblk->cx = bgroupcx;
    currblk->flags = flags|PM_ENDLIST;

    if( begingroup ) {
        currblk->flags |= begingroup;
        begingroup = 0;
    }
}

/* _removeundoinfo() searches for a specified frame by its offset pointer.
   When the pointer is to be found in the undo memory, currblk is set
   to that entry in the undo buffer, so this entry and the following
   entries are never used again... */
static void _removeundoinfo( unsigned long ptr ) {

    struct memblk *mb;

    for( mb = currblk; !(mb->flags & PM_BEGINLIST); mb = _prevblk(mb) );
    for( ;mb != currblk; mb = _nextblk(mb) ) {
        if( mb->ptr == ptr ) {
            mb = _prevblk(mb);
            currblk = mb;
            SETBIT( currblk->flags, PM_ENDLIST );
            return;
        }
    }
}

/* m_isrestoreable() returns a BOOL value indicating if the current
   contents of the undo buffer allows undoing an editing command */
BOOL m_isrestoreable( void ) {

    if( !(currblk->flags & PM_BEGINLIST) || (ed.Flags & LINECHANGED) )
        return TRUE;
    return FALSE;
}

BOOL m_isredoable( void ) {

    if( !(currblk->flags & PM_ENDLIST) && !(ed.Flags & LINECHANGED) )
        return TRUE;
    return FALSE;
}

void m_beginundogroup( void ) {

    begingroup = PM_BEGINGROUP;

    if( !(ed.Flags & LINECHANGED) )
        bgroupcx = ed.CursorX;
}

void m_endundogroup( void ) {

    SETBIT(currblk->flags,PM_ENDGROUP);
    begingroup = 0;
}

static struct memblk *_nextblk(struct memblk *mb){

    if( mb == &ring[RINGMEMSIZE-1] )
        return ring;
    return &mb[1];
}

static struct memblk *_prevblk(struct memblk *mb){

    if( mb == ring )
        return &mb[RINGMEMSIZE-1];
    return &mb[-1];
}

void m_undo( void ) {

    unsigned long _huge *desc;
    int grouplevel = 0;
    unsigned long swap;

    if( !m_putcurrentline() ) return;
    ++restoring;

    while( !(currblk->flags & PM_BEGINLIST) ) {

        if( currblk->flags & PM_BEGINGROUP )
            --grouplevel;
        if( currblk->flags & PM_ENDGROUP )
            ++grouplevel;

        desc = mh.desc - currblk->line;

        swap = *desc;

        if( currblk->flags & PM_INSERTLINE ) {
            m_deleteline( currblk->line, 1 );
        } else {
            if( currblk->flags & PM_DELETELINE ) {
                m_insertline( currblk->line );
            }

            /* mark old frame as free */
            if( *desc != NULLPTR )
               *(*desc+mh.base) = FREEFRAME|SMALLFRAME;

            /* put new frame pointer */
            if( currblk->ptr != NULLPTR ) {
                *desc = currblk->ptr;
                *(*desc+mh.base) = ALLOCFRAME|SMALLFRAME;
            }

            CLEARBIT( ed.Flags, LINECHANGED );
            SETBIT( ed.Flags, MODIFIED );
        }

        currblk->ptr = swap;

        e_goto( ed.WindowX, ed.WindowY, currblk->cx, currblk->line );

        /* get previous currblk */
        currblk = _prevblk( currblk );

        if( grouplevel <= 0 )
            break;
    }

    m_getcurrentline();
    --restoring;
}

void m_redo( void ) {

    unsigned long _huge *desc;
    int grouplevel = 0;
    unsigned long swap;

    ++restoring;
    if( !m_putcurrentline() ) return;

    while( !(currblk->flags & PM_ENDLIST) ) {

        /* get next currblk */
        currblk = _nextblk(currblk);

        if( currblk->flags & PM_BEGINGROUP )
            ++grouplevel;
        if( currblk->flags & PM_ENDGROUP )
            --grouplevel;

        desc = mh.desc - currblk->line;

        swap = *desc;

        if( currblk->flags & PM_DELETELINE ) {
            m_deleteline( currblk->line, 1 );
        } else {
            if( currblk->flags & PM_INSERTLINE ) {
                m_insertline( currblk->line );
            }

            /* mark old frame as free */
            if( *desc != NULLPTR )
               *(*desc+mh.base) = FREEFRAME|SMALLFRAME;

            /* put new frame pointer */
            if( currblk->ptr != NULLPTR ) {
                *desc = currblk->ptr;
                *(*desc+mh.base) = ALLOCFRAME|SMALLFRAME;
            }

            CLEARBIT( ed.Flags, LINECHANGED );
            SETBIT( ed.Flags, MODIFIED );
        }

        currblk->ptr = swap;

        e_goto( ed.WindowX, ed.WindowY, currblk->cx, currblk->line );

        if( grouplevel <= 0 )
            break;
    }

    m_getcurrentline();
    --restoring;
}

/*
 *    Name: m_putmsg
 *  Return: length of written string
 * Purpose: Prints out a message on the status line
 *
 *
 *
 * (c)M.Schollmeyer
 */
int m_putmsg( unsigned int reason, char *fmt, ... ) {

    va_list marker;
    int len = 0, i;
    char buffer[SCREEN_WIDTH+1];
    unsigned int color = CR_STATUS;

    contstat = reason;

    if( verbose > 0 ) {
        if( (reason == MSG_ERROR) || (reason==MSG_TEXERR) ) color = CR_STATUSERROR;
        va_start( marker, fmt );
        len = vsprintf( buffer, fmt, marker );
        for( i = len; i < VideoBase->ScreenWidth; ++i )
            buffer[i] = ' ';
        buffer[i] = '\0';
        HideMouse();
        VideoStr( color, buffer, 0, (unsigned char)(VideoBase->ScreenHeight - 1) );
        ShowMouse();
    }
    return len;
}

/*
 *    Name: putxmsg
 *  Return: length of written string
 * Purpose: Prints out a message on the status line at a specified position
 *
 *
 *
 * (c)M.Schollmeyer
 */
int m_putxmsg( unsigned int attr, int x, char *fmt, ... ) {

    va_list marker;
    int len = 0;
    char buffer[EDIT_WIDTH];

    if( verbose > 0 ) {
        va_start( marker, fmt );
        len = vsprintf( buffer, fmt, marker );
        HideMouse();
        VideoStr( attr, buffer, x, (unsigned char)(VideoBase->ScreenHeight - 1) );
        ShowMouse();
    }
    return len;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
struct MarkRegion *m_getmr( void ) {

    static struct MarkRegion mr;
    int len;

    mr.buffer[0] = '\0';
    mr.flags = 0;

    /* setup mark region */
    if( ed.Flags & MARKED ) {
        SETBIT( mr.flags, MR_MARKED );

        if( ed.MarkY < ed.CursorY ) {
            SETBIT( mr.flags, MR_MULTILINE|MR_FORWARDS );

            mr.UpperY = ed.MarkY;
            mr.UpperX = ed.MarkX;
            mr.LowerY = ed.CursorY;
            mr.LowerX = ed.CursorX;
        } else if( ed.MarkY == ed.CursorY ) {
            mr.UpperY = mr.LowerY = ed.MarkY;
            if( ed.MarkX < ed.CursorX ) {
                SETBIT( mr.flags, MR_FORWARDS );

                mr.UpperX = ed.MarkX;
                mr.LowerX = ed.CursorX;
            } else {
                mr.UpperX = ed.CursorX;
                mr.LowerX = ed.MarkX;
            }

            len = strlen( currline );
            if( len > mr.UpperX ) {
                len = mr.LowerX-mr.UpperX;
                strncpy( mr.buffer, currline+mr.UpperX, len );
                mr.buffer[len] = '\0';
            }
        } else {
            SETBIT( mr.flags, MR_MULTILINE );

            mr.LowerY = ed.MarkY;
            mr.LowerX = ed.MarkX;
            mr.UpperY = ed.CursorY;
            mr.UpperX = ed.CursorX;
        }
    } else {
        SETBIT( mr.flags, MR_MULTILINE );

        mr.UpperY = mr.UpperX = 0;
        mr.LowerY = mh.numlines;
        mr.LowerX = EDIT_WIDTH-1;
    }

    return &mr;
}



/* end of file memory.c */
